/*****************************************************************
 * timesheet.c
 * The Palmpilot Timesheet application.
 * Copyright (C) 1999 Stuart Nicholson
 *
 * -------------------------------------------------------------
 * Update: from v1.5.3 to 'version 1.5.4'; 12 to 14-8-2005.
 * [jo] Updated (fixed, to a degree) by Johan van Schalkwyk,
 *  also from New Zealand! Everyone loved the program, 
 *  but it wouldn't run on my Pilot/emulator so I 'fixed' it!
 *
 * Stuart (1999) did a bloody good job, and it's a beautiful program.
 * Palmos 'changed the rules' a bit, so that their structures,
 * which always were a little opaque, now resulted in breaks
 * in Stuart's program. He also did a few mildly naughty things
 * like reading past some buffer sources. There were a few date
 * string problems too. A lot of the issues are founded in bad
 * design of PalmOS. (But still better than MS :-)
 * I've also changed some `ID numbers' like groupID_tsDayGroup
 * as they should be UInt8. I have indulged in some gratuitous
 * fiddling, like changing Char[n] to calls to my memory-allocation
 * function e_New with a corresponding e_Delete at the end.
 * I have removed upgrading of versions under 1.5.3, as
 * I strongly doubt anyone is still using these!
 *
 * Changes are usually annotated with the handle [jo].
 * Where I've not put [jo] you'll find my comments thus: /// 
 * I generally do NOT amend the old-style Palm Pilot C code.
 * I've lumped together all of the C components into one long
 * 6K-line program, simply because it made my task easier (not
 * wandering around within lots of little files), and simplified
 * my re-creation of the makefile. If you still believe in the
 * myth of reusable code, you might recreate the components!
 * I haven't done anything too wicked in the makefile (or if I
 * have, I'm too ignorant to realise this). I took out the
 * awk stuff, so you'll have to manually update the date.
 * 
 * I'm afraid I did everything using WinEdt and compiled using
 * GCC under CYGWIN, with testing on POSE Palm Emulator. 
 * GDB was used to locate problems. 
 *
 * STILL TO DO:
 * 1. Fix memory leaks. [d] below seems to address this! Are there more?
 * -------------------------------------------------------------
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 ******************************************************************/

/*****************************************************************
 * [Thus spake Stuart:]
 *
 * This application represents my first attempt at Palmpilot
 * programming using the GNU Pilot SDK and I begin coding it in an
 * attempt to understand the Pilot GUI and Event model.
 *
 * As such, much of the code below is rather hacky and unclear
 * (particularly those sections that deal with the GUI and moving
 * between Pilot forms).
 ******************************************************************/

/******************************************************************
 * PalmOS includes.
 ******************************************************************/
#include <PalmCompatibility.h> // [jo] my little fix for PalmOS.
#include <PalmOS.h>

/******************************************************************
 * Application includes.
 ******************************************************************/

#include "Timesheet.h"
#include "Globals.h"
#include "callback.h"
#include "resource.h"

/******************************************************************
 * Jo's stuff: [jo]
 ******************************************************************/
 
#define NINETEENTWENTY 1920
// [jo] ok I was playing.

// [jo] my own little functions at the end of everything else. 
Int16 e_Delete (MemPtr memP);
MemPtr e_New ( Int16 memsize); 
Int16 xFill (Char * p0, Int16 slen, Char c);
Int16 xCopy (Char * dest, Char * xsrc, Int16 maxlen);
void SHOWERROR(const Char * msg, Int32 nmbr);

// ---------------------------------------------------------------//
// [jo] Headers for all of the amalgamated code sections below. 

#include "AboutForm.h"
#include "Category.h"
#include "Database.h"
#include "DetailForm.h"
#include "EditCatEntryForm.h"
#include "EditCatForm.h"
#include "MainForm.h"
#include "PrefForm.h"
#include "UIUtil.h"
#include "WeeklyForm.h"


/******************************************************************
 * Application Function Prototypes
 ******************************************************************/

void AppEventLoop(void);
Err StartApplication(void);
Err RomVersionCompatible(DWord requiredVersion, Word launchFlags);
void StopApplication(void);
DWord PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags);

Boolean MainFormEventHandler(EventPtr event);
void MainFormGotoWeekday(Byte dayButton, Boolean refreshGUI);
void MainFormChangeDate(Int deltaDays, Boolean refreshGUI);
void MainFormNewEntryAttempt(void);
UInt MainFormErasePriorTo(DateType * priorToDate);
void MainFormGotoDay(void);
void MainFormGotoRecord(UInt recIdx);
void MainFormGotoEarliestDay(void);
void MainFormGotoLatestDay(void);
void MainFormInit(void);
void MainFormRefresh(void);
void MainFormTableDrawCell(VoidPtr tablePtr, Word row, Word column, RectanglePtr bounds);
void MainFormTableRowTally(UInt recIdx,
                                  TSDurationRecType* totalDuration,
                                  TSDurationRecType* totalChargeableDuration );
void MainFormTableDraw(void);
void MainFormTableScroll(Byte dir);
void MainFormScrollButtonUpdate(FormPtr formPtr);


/******************************************************************
 * Application global variables
 ******************************************************************/

/* Application copy of the relevant system preferences. */
DateFormatType TSSysDateFormat;
DateFormatType TSSysLongDateFormat;
Byte TSSysWeekStartDay;

/* Application copy of the preferences stored in the database record 0. */
TSAppPrefType TSAppPrefs;

/* Single global entry timer is supported */
TSEntryTimerType TSEntryTimer;

/* The presently open timesheet database. */
DmOpenRef TSDatabase = NULL;

/* True if a record exists for the present day. */
Boolean TSPresentDayRecExists = false;

/* The day record index for the present day (if TSPresentDayRecExists = true),
 * otherwise the record index to insert a day record for the present day.
 */
UInt TSPresentDayRecIdx = TSFirstDayRecIdx;

/******************************************************************
 * Globals relating to the GUI and the presently visible form.
 ******************************************************************/
/* Used to pass information to the Details form. Set this to the database record idx of the entry
 * you're editing, or to 0 if you're attempting to create a *new* entry. */
UInt TSEditEntryIdx = 0;

/* The database record number of the list to edit in the category Edit dialog */
Byte TSEditCatRecIdx = 0;

/* The category entry number of the category beging edited in the category entry Edit dialog, or 0 if you're
 * creating a new category. */
Byte TSEditCatEntryIdx = 0;

/* The present logical day button that is selected on the main form (ie. 0 to 6, not the actual button ID) */
Byte TSSelectedDayButton;

/* The day entry that presently appears at the top of the table in the current form (Main, Weekly or Monthly) */
SWord TSTableTopEntry;

/* The date of the present day being displayed. */
DateType TSPresentDayDate;

/* The adjusted summary date (based on TSPresentDayDate) being displayed. */
DateType TSPresentSummaryDate;



/******************************************************************
 * Event handler for the main form.
 ******************************************************************/
Boolean MainFormEventHandler (EventPtr event)
{
  DateType priorToDate;
  SWord day;
  SWord month;
  SWord year;
  UInt priorToDays;
  VoidHand hand;
  FormPtr formPtr;
  /// Char priorTo[dowLongDateStrLength]; /// noooo, not 20+1. best make this dowLongDateStrLength
  // [jo] Yep. that fixed the bus error 13-8-2005.
  Char * priorTo; ///
  Boolean handled = false;

  priorTo = e_New(dowLongDateStrLength); /// [jo] my fiddling with memory allocation

  if (event->eType == ctlSelectEvent)
    {
      /* a control button was clicked */
      if (event->data.ctlSelect.controlID >= buttonID_tsDay0 && event->data.ctlSelect.controlID <= buttonID_tsDay6)
        {
          /* one of the day buttons was selected */
          MainFormGotoWeekday(event->data.ctlSelect.controlID - buttonID_tsDay0, true);
          handled = true;
        }
      else
        {
          /* some other button or control was selected */
          if (event->data.ctlSelect.controlID == buttonID_tsNextWeek)
            {
              /* select the next week */
              MainFormChangeDate(7, true);
              handled = true;
            }
          else if (event->data.ctlSelect.controlID == buttonID_tsPrevWeek)
            {
              /* select the previous week */
              MainFormChangeDate(-7, true);
              handled = true;
            }
          else if (event->data.ctlSelect.controlID == buttonID_tsNew)
            {
              /* attempt create a new day entry  */
              MainFormNewEntryAttempt();
              handled = true;
            }
          else if (event->data.ctlSelect.controlID == buttonID_tsGoto)
            {
              /* jump to user specified date */
              day = TSPresentDayDate.day;
              month = TSPresentDayDate.month;
              year = TSPresentDayDate.year + NINETEENTWENTY;
              hand = DmGet1Resource('tSTR', stringID_tsSelectDate);
              if (SelectDay(selectDayByDay, &month, &day, &year, (CharPtr) MemHandleLock(hand)))
                {
                  /* user has entered date to go to, save the new date */
                  TSPresentDayDate.day = day;
                  TSPresentDayDate.month = month;
                  TSPresentDayDate.year = year - NINETEENTWENTY;
                  MainFormGotoDay();
                  MainFormTableDraw();
                }
              MemHandleUnlock(hand);
              DmReleaseResource(hand);
              handled = true;
            }
          else if (event->data.ctlSelect.controlID == buttonID_tsGotoTimed)
            {
              /* User wants to go to timed entry in database */
              if (TSEntryTimer.recIdx > 0 && TSEntryTimer.secs > 0)
              {
                /* Go to the timed entry */
                MainFormGotoRecord(TSEntryTimer.recIdx);
              }
              else
              {
                /* No entry being timed */
                FrmAlert(alertID_tsNoTimedEntry);
              }
            }
          else if (event->data.ctlSelect.controlID == buttonID_tsWeekViewButton)
            {
              /* Change to the weekly summary view */
              FrmGotoForm(formID_tsWeekly);
              handled = true;
            }
        }
    }
  else if (event->eType == ctlRepeatEvent)
    {
      if (event->data.ctlSelect.controlID == buttonID_tsTableUp)
        {
          /* scroll table up */
          MainFormTableScroll(pageUpChr);
          MainFormTableDraw();
        }
      else if (event->data.ctlSelect.controlID == buttonID_tsTableDown)
        {
          /* scroll table down */
          MainFormTableScroll(pageDownChr);
          MainFormTableDraw();
        }
    }

  else if (event->eType == menuEvent)
    {
      // a menu item has been selected 
      if (event->data.menu.itemID == menuitemID_tsGotoEarliest)
        {
          // user wants to go to earliest day in database 
          MainFormGotoEarliestDay();
        }
      else if (event->data.menu.itemID == menuitemID_tsGotoLatest)
        {
          // user wants to go to latest day in database 
          MainFormGotoLatestDay();
        }
      else if (event->data.menu.itemID == menuitemID_tsEraseDay)
        {
          // user wants to erase all entries in the current day 
          if (TSPresentDayRecExists)
            {
              // there's at least one entry that can be deleted for today 
              hand = DmGet1Resource('tSTR', stringID_tsDay);
              if (FrmCustomAlert(alertID_tsConfirmDayWeekDel,
                  (CharPtr)MemHandleLock(hand), NULL, NULL) == 0)
                {
                  // user confirms delete, delete all entries for today 
                  while (DetailFormDeleteEntry(TSPresentDayRecIdx + 1) != 0)
                    {
                      // body intentionally empty 
                    }
                  // nothing to display, move table to top row 
                  TSTableTopEntry = 0;
                  // Main form has changed 
                  MainFormRefresh();
                }
              MemHandleUnlock(hand);
              DmReleaseResource(hand);
            }
          else
            {
              // nothing to do! no entries for today 
              hand = DmGet1Resource('tSTR', stringID_tsDay);
              FrmCustomAlert(alertID_tsEmptyDayWeek, (CharPtr)MemHandleLock(hand), NULL, NULL);
              MemHandleUnlock(hand);
              DmReleaseResource(hand);
            }
          handled = true;
        }
      else if (event->data.menu.itemID == menuitemID_tsErasePrior)
        {
          // user wants to erase all entries prior to a certain date 
          day = TSPresentDayDate.day;
          month = TSPresentDayDate.month;
          year = TSPresentDayDate.year + NINETEENTWENTY;
          hand = DmGet1Resource('tSTR', stringID_tsErasePriorTo);
          if (SelectDay(selectDayByDay, &month, &day, &year, (CharPtr) MemHandleLock(hand)))
            {
              // user has entered date to purge prior to, make it so... 
              DateToDOWDMFormat(month, day, year,
                                (TSAppPrefs.prefFlags & TSPrefShortDateFlag) ? TSSysDateFormat : TSSysLongDateFormat,
                                priorTo);
              if (FrmCustomAlert(alertID_tsConfirmPrior, priorTo, NULL, NULL) == 0)
                {
                  // user has confirmed prior to erase...do it 
                  priorToDate.day = day;
                  priorToDate.month = month;
                  priorToDate.year = year - NINETEENTWENTY;
                  priorToDays = DateToDays(priorToDate);
                  if (MainFormErasePriorTo(&priorToDate) > 0)
                    {
                      // have actually purged some records, must update globals and form 
                      FindDayRec(&TSPresentDayDate, &TSPresentDayRecIdx, &TSPresentDayRecExists);
                      // check if we need to re-draw main table 
                      if (priorToDays >= DateToDays(TSPresentDayDate))
                        {
                          // the current day has been erased, move table to top row 
                          TSTableTopEntry = 0;
                          MainFormRefresh();
                        }
                    }
                }
            }
          MemHandleUnlock(hand);
          DmReleaseResource(hand);
          handled = true;
        }
      else if (event->data.menu.itemID == menuitemID_tsPref)
        {
          // display the prefs form 
          FrmPopupForm(formID_tsPref);
          handled = true;
        }
      else if (event->data.menu.itemID == menuitemID_tsAbout)
        {
          // display the About form 
          FrmPopupForm(formID_tsAbout);
          // FrmHelp(helpID_tsAbout); 
          handled = true;
        }
    }

  else if (event->eType == tblEnterEvent)
    {
      /* a row in the main table has been selected, note I've used tblEnterEvents instead of tblSelectEvents so the
       * user doesn't get to see the annoying GUI 'item inversion' behaviour, which just looks plain icky. */
      SndPlaySystemSound(sndClick);
      /* set the record index of the entry we want to edit, depending on what record currently appears at the
       * top of the table. */
      TSEditEntryIdx = TSPresentDayRecIdx + TSTableTopEntry + event->data.tblSelect.row + 1;
      /* show the details form to the user */
      FrmPopupForm(formID_tsDetail);
      handled = true;
    }
  else if (event->eType == keyDownEvent)
    {
      /* key has been pressed, is it one app should respond to? */
      if (event->data.keyDown.chr == pageUpChr)
        {
          /* command key up, goto previous day */
          /* go back a day */
          MainFormChangeDate(-1, true);
          handled = true;
        }
      else if (event->data.keyDown.chr == pageDownChr)
        {
          /* command key down, goto next day */
          MainFormChangeDate(1, true);
          handled = true;
        }
    }
  else if (event->eType == frmGotoEvent)
    {
      formPtr = FrmGetActiveForm();
      /* initialise the main form to today */
      MainFormInit();
      /* draw the main form */
      FrmDrawForm(formPtr);
      MainFormGotoDay();
      MainFormRefresh();
      /* Go to a specific record on the main form, req'd for Find functionality */
      MainFormGotoRecord(event->data.frmGoto.recordNum);
      handled = true;
    }
  else if (event->eType == frmOpenEvent)
    {
      /* the Main form has been opened, this will only occur once
       * each time the application is opened, so it's safe to
       * initialise the main form here. */
      formPtr = FrmGetActiveForm();
      /* initialise the main form to today */
      MainFormInit();
      /* draw the main form */
      FrmDrawForm(formPtr);
      MainFormGotoDay();
      MainFormRefresh();
      handled = true;
    }
  e_Delete(priorTo);
  return handled;
}


/******************************************************************
 * Run the application event loop.
 ******************************************************************/
void
AppEventLoop(void)
{
  Err error;
  int formID;
  FormPtr formPtr;
  EventType event;

  do
    {
      /* get the next event to handle */
      EvtGetEvent(&event, 50);
      /* let the system have first look at the event */
      if (SysHandleEvent(&event))
        continue;
      if (MenuHandleEvent((void *) 0, &event, &error))
        continue;
      /* handle form loads */
      if (event.eType == frmLoadEvent)
        {
          formID = event.data.frmLoad.formID;
          formPtr = FrmInitForm(formID);
          FrmSetActiveForm(formPtr);
          if (formID == formID_tsMain)
            {
              /* set up the handler for the main form */
              FrmSetEventHandler(formPtr, (FormEventHandlerPtr) MainFormEventHandler);
            }
          else if (formID == formID_tsWeekly)
            {
              /* Set up the handler for the weekly summary form */
              FrmSetEventHandler(formPtr, (FormEventHandlerPtr) WeeklyFormEventHandler);
            }
          else if (formID == formID_tsDetail)
            {
              /* set up the handler for the details form */
              FrmSetEventHandler(formPtr, (FormEventHandlerPtr) DetailFormEventHandler);
            }
          else if (formID == formID_tsEditCat)
            {
              /* set up the handler for the edit form */
              FrmSetEventHandler(formPtr, (FormEventHandlerPtr) EditCatFormEventHandler);
            }
          else if (formID == formID_tsEditCatEntry)
            {
              /* set up the handler for the category edit form */
              FrmSetEventHandler(formPtr, (FormEventHandlerPtr) EditCatEntryFormEventHandler);
            }
          else if (formID == formID_tsPref)
            {
              /* set up the handler for the preferences edit form */
              FrmSetEventHandler(formPtr, (FormEventHandlerPtr) PrefFormEventHandler);
            }
          else if (formID == formID_tsAbout)
            {
              /* set up the handler for the about form */
              FrmSetEventHandler(formPtr, (FormEventHandlerPtr) AboutFormEventHandler);
            }
        }
      FrmDispatchEvent(&event);
    }
  while (event.eType != appStopEvent);
}

/***********************************************************************
 * Checks that Pilot ROM version meets minimum requirement.
 * Returns error code or zero if rom is compatible
 ***********************************************************************/
Err
RomVersionCompatible(DWord requiredVersion, Word launchFlags)
{
  DWord romVersion;

  /* see if we're on in minimum required version of the ROM or later. */
  FtrGet(sysFtrCreator, sysFtrNumROMVersion, &romVersion);
  if (romVersion < requiredVersion)
    {
      if ((launchFlags & (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp)) ==
          (sysAppLaunchFlagNewGlobals | sysAppLaunchFlagUIApp))
        {
          FrmAlert(alertID_tsRomIncompat);
          /* Pilot 1.0 will continuously relaunch this app unless we switch to another safe one. */
          if (romVersion < 0x02000000)
            {
///              Err err; // [jo] unused.
              AppLaunchWithCommand(sysFileCDefaultApp, sysAppLaunchCmdNormalLaunch, NULL);
            }
        }
      return sysErrRomIncompatible;
    }
  /* rom version okay */
  return 0;
}

/******************************************************************
 * Start the application, opening databases and selecting initial forms etc.
 ******************************************************************/
Err
StartApplication(void)
{
  Err error;
  SystemPreferencesType sysPrefs;
  DateTimeType now;

  /* Open or create the application database */
  error = OpenDatabase();
  if (error)
    {
      /* Failed to open database */
      return error;
    }
  /* Get a copy of the system's preferences */
  PrefGetPreferences(&sysPrefs);
  /* Save the system preferences we're interested in */
  TSSysDateFormat = sysPrefs.dateFormat;
  TSSysLongDateFormat = sysPrefs.longDateFormat;
  TSSysWeekStartDay = sysPrefs.weekStartDay;
  /* Set the dateTime to today */
  TimSecondsToDateTime(TimGetSeconds(), &now);
  /* Convert DateTime to Date */
  TSPresentDayDate.year = now.year;
  TSPresentDayDate.month = now.month;
  TSPresentDayDate.day = now.day;
  /* Application started okay */
  return 0;
}

/******************************************************************
 * Stop the application, closing any open databases.
 ******************************************************************/
void
StopApplication(void)
{
  LocalID summaryID;
  Int summaryCardNum;
  DmOpenRef summaryDB;

  /* Close the database */
  ClosePreferences();
  DmCloseDatabase(TSDatabase);
  TSDatabase = NULL;
  /* Check if there's an existing summary database */
  summaryDB = DmOpenDatabaseByTypeCreator(TSSummaryDBType, TSAppID, dmModeReadWrite);
  if (summaryDB)
    {
      /* Get locaID, cardID etc. so we can delete database */
      DmOpenDatabaseInfo(summaryDB, &summaryID, NULL, NULL, &summaryCardNum, NULL);
      /* Close and delete existing summary database */
      DmCloseDatabase(summaryDB);
      DmDeleteDatabase(summaryCardNum, summaryID);
    }
  /// [jo] 14-8-2005 Added the following to keep memory clean [d]
  FrmEraseForm(FrmGetActiveForm());
  FrmDeleteForm(FrmGetActiveForm()); /// free up current form.

}

/******************************************************************
 * Palmpilot main function, entry point for the application.
 ******************************************************************/
DWord
PilotMain(Word cmd, Ptr cmdPBP, Word launchFlags)
{
  Word error;
  EventType event;

  if (cmd == sysAppLaunchCmdNormalLaunch)
    {
      /* Check the ROM version is PalmOS 2.0 or better */
      error = RomVersionCompatible(TSRomVersion, launchFlags);
      if (error)
        {
          /* App won't run, wrong ROM version */
          return error;
        }
      else
        {
          /* Run app */
          error = StartApplication();
          if (error)
            {
              /* Failed to start application */
              return error;
            }
          /* Go to the application's initial form */
          FrmGotoForm(formID_tsMain);
          AppEventLoop();
          StopApplication();
        }
    }
  else if (cmd == sysAppLaunchCmdFind)
    {
      FindInDatabase((FindParamsPtr) cmdPBP);
    }
  else if (cmd == sysAppLaunchCmdGoTo)
    {
      /* If the user wants to goto a record listed in the find dialog it sends the following notification.
         Check if new global variables are set up.  This occurs when the system intends for the app to start */
      if (launchFlags & sysAppLaunchFlagNewGlobals)
        {
          /* This occurs when the user wants to see a record in this application's database.
             The app must be started and a view to display the data must be opened */
          /* check the ROM version is PalmOS 2.0 or better */
          error = RomVersionCompatible(TSRomVersion, launchFlags);
          if (error)
            {
              /* App won't run, wrong ROM version */
              return error;
            }
          else
            {
              /* Run app */
              error = StartApplication();
              if (error)
                {
                  /* Failed to start application */
                  return error;
                }
            }
        }
      else
        {
          /* The app was already running, close the existing UI */
          FrmCloseAllForms();
        }
      /* Load the application's main form */
      MemSet(&event, 0, sizeof(EventType));
      event.eType = frmLoadEvent;
      event.data.frmLoad.formID = formID_tsMain;
      EvtAddEventToQueue(&event);
      /* Go to the application's main form */
      event.eType = frmGotoEvent;
      event.data.frmGoto.formID = formID_tsMain;
      event.data.frmGoto.recordNum = ((GoToParamsPtr) cmdPBP)->recordNum;
      EvtAddEventToQueue(&event);
      /* If the app is not already open, run the app */
      if (launchFlags & sysAppLaunchFlagNewGlobals)
        {
          AppEventLoop();
          StopApplication();
        }
    }
  /* App ran normally, or ignored launch code */
  return 0;
}

/* End of section! */

// ============================================================================ //
// MainForm.c:
// #include "MainForm.c"

/******************************************************************
 * Module internal function prototypes
 ******************************************************************/
// [moved up]

/******************************************************************
 * Erase all existing timesheet entry records and day records PRIOR
 * to the specified date.
 ******************************************************************/
UInt
MainFormErasePriorTo(DateType * priorToDate)
{
  UInt idx;
  UInt priorToDays;
  UInt numEntries;
  UInt recDays;
  UInt recDelCount;
  VoidHand hand;
  TSDayRecType *dayPtr;

  /* Convert the prior to date to days */
  priorToDays = DateToDays(*priorToDate);
  recDelCount = 0;
  /* Work until we run out of day records */
  while (DmNumRecords(TSDatabase) > TSFirstDayRecIdx)
    {
      hand = DmQueryRecord(TSDatabase, TSFirstDayRecIdx);
      dayPtr = MemHandleLock(hand);
      /* convert the record date to days */
      recDays = DateToDays(dayPtr->dayDate);
      numEntries = dayPtr->numEntries;
      MemHandleUnlock(hand);
      /* delete this record? */
      if (recDays < priorToDays)
        {
          /* yes, delete day record */
          DmRemoveRecord(TSDatabase, TSFirstDayRecIdx);
          /* Deleting all entries for day, as well as day record itself */
          recDelCount += numEntries + 1;
          /* and delete all day entries */
          for (idx = 0; idx < numEntries; idx++)
            {
              DmRemoveRecord(TSDatabase, TSFirstDayRecIdx);
            }
        }
      else
        {
          /* no, stop searching */
          break;
        }
    }
  /* Check if we've just deleted the timer entry record (if any) */
  if (TSEntryTimer.recIdx > 0)
  {
    if (TSFirstDayRecIdx + recDelCount >= TSEntryTimer.recIdx)
    {
      /* Stop timing as the timed record has been deleted */
      TSEntryTimer.recIdx = 0;
      TSEntryTimer.secs = 0;
    }
    else
    {
      /* Adjust the timed record index accordingly */
      TSEntryTimer.recIdx -= recDelCount;
    }
  }
  return recDelCount;
}


/******************************************************************
 * Go to the current date, modifying application internal variables
 * as well as the Main Form controls appropriately.
 ******************************************************************/
void
MainFormGotoDay(void)
{
  int dayButton;
  int weekDay;

  /* set the appropriate main form day button for today */
  weekDay = DayOfWeek(TSPresentDayDate.month, TSPresentDayDate.day, TSPresentDayDate.year + NINETEENTWENTY);
  if (TSSysWeekStartDay == 0)
    {
      /* week starts Sunday */
      dayButton = weekDay;
    }
  else
    {
      /* week starts Monday */
      dayButton = (weekDay == 0) ? (6) : (weekDay - 1);
    }
  FrmSetControlGroupSelection(FrmGetActiveForm(), (UInt8) groupID_tsDayGroup, buttonID_tsDay0 + dayButton); ///
  TSSelectedDayButton = dayButton;
  /* find the day record for today (if any) */
  FindDayRec(&TSPresentDayDate, &TSPresentDayRecIdx, &TSPresentDayRecExists);
  /* refresh the main form date */
  FormUpdateDay(&TSPresentDayDate); // gave problems. fixed [jo]
}

/******************************************************************
 *
 ******************************************************************/
void
MainFormGotoRecord(UInt recIdx)
{
  VoidHand recHand;
  TSDayRecType *dayPtr;
  UInt dayRecIdx;

  /* start fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_PROLOGUE;

  /* Have a record number, need to find day record that contains entry record number. */
  dayRecIdx = FindPrevDayRecIdx(TSDatabase, recIdx);
  recHand = DmQueryRecord(TSDatabase, dayRecIdx);
  dayPtr = MemHandleLock(recHand);
  TSPresentDayDate.day = dayPtr->dayDate.day;
  TSPresentDayDate.month = dayPtr->dayDate.month;
  TSPresentDayDate.year = dayPtr->dayDate.year;
  MemHandleUnlock(recHand);
  /* Refresh display */
  MainFormGotoDay();
  /* Open the specified entry */
  TSEditEntryIdx = recIdx;
  /* show the details form to the user */
  FrmPopupForm(formID_tsDetail);

  /* end fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_EPILOGUE;
}

/******************************************************************
 * User has clicked one of the week-day buttons on the main form to
 * change the day of the present week.
 ******************************************************************/
void MainFormGotoEarliestDay(void)
{

  /* Find the earliest day rec (if one exists) */
  if (FindEarliestDayRec())
    {
      /* go to the earliest day, updating GUI and internal globals */
      MainFormGotoDay();
      MainFormTableDraw();
    }
  else
    {
      /* no earlier day to go to, alert user */
      FrmAlert(alertID_tsEmptyDB);
    }
}

/******************************************************************
 * User has clicked one of the week-day buttons on the main form to
 * change the day of the present week.
 ******************************************************************/
void
MainFormGotoLatestDay(void)
{
  if (FindLatestDayRec())
    {
      /* go to the latest day, updating GUI and internal globals */
      MainFormGotoDay();
      MainFormTableDraw();
    }
  else
    {
      /* no later day to go to, alert user */
      FrmAlert(alertID_tsEmptyDB);
    }
}

/******************************************************************
 * User has clicked one of the week-day buttons on the main form to
 * change the day of the present week.
 ******************************************************************/
void
MainFormGotoWeekday(Byte dayButton, Boolean refreshGUI)
{
  Int deltaDays;

  /* check if the day has really changed */
  if (dayButton == TSSelectedDayButton)
    {
      /* nope, do nothing */
      return;
    }
  else
    {
      /* calculate the CHANGE in days depending on the week day button selected */
      deltaDays = dayButton - TSSelectedDayButton;
      /* save the new day button selected */
      TSSelectedDayButton = dayButton;
      /* modify the date accordingly */
      MainFormChangeDate(deltaDays, refreshGUI);
    }
}

/******************************************************************
 * Change the date of the day presently being displayed, will also
 * update the GUI appropriately if refreshGUI is set to true.
 ******************************************************************/
void
MainFormChangeDate(Int deltaDays, Boolean refreshGUI)
{
  Int weekDay;

  /* modify the date accordingly */
  DateAdjust(&TSPresentDayDate, deltaDays);
  /* find the day record for today (if any) */
  FindDayRec(&TSPresentDayDate, &TSPresentDayRecIdx, &TSPresentDayRecExists);
  /* refresh the main form if req'd */
  if (refreshGUI)
    {
      weekDay = DayOfWeek(TSPresentDayDate.month, TSPresentDayDate.day, TSPresentDayDate.year + NINETEENTWENTY);
      if (TSSysWeekStartDay == 0)
        {
          /* week starts Sunday */
          TSSelectedDayButton = weekDay;
        }
      else
        {
          /* week starts Monday */
          TSSelectedDayButton = (weekDay == 0) ? (6) : (weekDay - 1);
        }
      FrmSetControlGroupSelection(FrmGetActiveForm(), (UInt8) groupID_tsDayGroup, (UInt16)buttonID_tsDay0 + TSSelectedDayButton); ///
      FormUpdateDay(&TSPresentDayDate);
      /* reset TSTableTopEntry now we've changed day */
      TSTableTopEntry = 0;
      MainFormTableDraw();
    }
}

/******************************************************************
 * Initialise the main form, which is about to be shown for the first time.
 ******************************************************************/
void
MainFormInit(void)
{
  Char chr;
  Word id;
  CharPtr str1, str2;
  FormPtr formPtr;

  /* If the start-day-of-week is monday rearrange the labels on the days-of-week push buttons.
   * Note: This code is lifted from the PalmPilot Datebook App, no point reinventing the wheel eh? */
  if (TSSysWeekStartDay == monday)
    {
      formPtr = FrmGetActiveForm();
      str1 = (Char *) CtlGetLabel( (ControlType *)FrmGetObjectPtr((FormType *)formPtr, (UInt16)(FrmGetObjectIndex((FormType *)formPtr, (UInt16)buttonID_tsDay0)))); ///
      chr = *str1;
      for (id = buttonID_tsDay1; id <= buttonID_tsDay6; id++)
        {
          str2 =(Char *)  CtlGetLabel(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, id))); ///
          *str1 = *str2;
          str1 = str2;
        }
      *str1 = chr;
    }
  /* Select the day view button */
  FrmSetControlGroupSelection(FrmGetActiveForm(), (UInt8) groupID_tsViewGroup, (UInt16) buttonID_tsDayViewButton); ///
  /* initialise the table globals */
  TSTableTopEntry = 0;
}

/******************************************************************
 * Must be called each time the Main form is returned to. Updates the
 * Main Form table and any other appropriate controls.
 ******************************************************************/
void MainFormRefresh(void)
{
  VoidHand recHand;
  BitmapPtr bitmapPtr;
  ControlPtr controlPtr;
  FormPtr formPtr;
  Coord jox;
  Coord joy; // my little hack. 

  /*** Get the active Main form ***/
  formPtr = FrmGetActiveForm();

  /*** Redraw the table ***/
  MainFormTableDraw();

  /*** Show or hide the timer button as appropriate ***/
  controlPtr = FrmGetObjectPtr( formPtr, FrmGetObjectIndex( formPtr, buttonID_tsGotoTimed ) );
  if ( TSEntryTimer.recIdx > 0 && TSEntryTimer.secs > 0 )
  {
    /*** Timed entry exists, show button ***/
    CtlShowControl( controlPtr );

    /*** Draw the bitmap ***/
    recHand = DmGet1Resource( 'Tbmp', bitmapID_tsStopWatch );
    bitmapPtr = MemHandleLock( recHand );
    controlPtr = FrmGetObjectPtr( formPtr, FrmGetObjectIndex( formPtr, bitmapID_tsStopWatch ) );

///    WinDrawBitmap( bitmapPtr, controlPtr->bounds.topLeft.x, controlPtr->bounds.topLeft.y );
    /// [jo] the above is strongly deprecated by PalmOS!
    /// ie. do NOT say controlPtr->bounds...
    /// Do we need to rewrite this [12-8-2005] to find the button bounds? Yep 
    /// 
  FrmGetObjectPosition(formPtr, FrmGetObjectIndex(formPtr, bitmapID_tsStopWatch), &jox, &joy);
  WinDrawBitmap( bitmapPtr, jox, joy );    
    MemHandleUnlock( recHand );
    DmReleaseResource( recHand );
  }
  else
  {
    /*** No time entry exists, hide button ***/
    CtlHideControl( controlPtr );
  }
}

/******************************************************************
 * User attempts to create a new timesheet entry for the present
 * day if possible.
 ******************************************************************/
void
MainFormNewEntryAttempt(void)
{
  VoidHand hnd;
  TSDayRecType *dayPtr;

  /* check if any new entry records can be inserted into the current day record */
  if (TSPresentDayRecExists)
    {
      /* get the record for the present day */
      hnd = DmQueryRecord(TSDatabase, TSPresentDayRecIdx);
      dayPtr = MemHandleLock(hnd);
      /* have a record for the present day, is it full? */
      if (dayPtr->numEntries >= TSMaxEntriesPerDay)
        {
          /* present day is full, tell the user and bail */
          MemHandleUnlock(hnd);
          FrmAlert(alertID_tsDayFull);
          return;
        }
      MemHandleUnlock(hnd);
    }                           /* else don't have a record for today, may have to create one later if the user saves their new entry */
  /* allow the user to enter details for the *new* record */
  TSEditEntryIdx = 0;           /* editing potentially *new* record */
  /* show the details form to the user */
  FrmPopupForm(formID_tsDetail);
}

/******************************************************************
 * Custom cell drawing routine for the MainForm table.
 ******************************************************************/
void
MainFormTableDrawCell(VoidPtr tablePtr, Word row, Word column, RectanglePtr bounds)
{
  /// Char txt[25 + 1];
  Char * txt;  
  Int idx;
  SWord txtWidth;
  SWord txtLen;
  Boolean txtFitsInWidth;
  FontID prevFont;
  VoidHand listHand;
  TSCatRecType *listPtr;
  RectangleType rowRect =
  {
    {0, 0},
    {159, 11}
  };                 /* warning: note these are hardcoded bounds! */
  
  /* start fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_PROLOGUE;
  txt = e_New(26);

  /* get the entry string index from the item */
  idx = TblGetItemInt(tablePtr, row, column);
  /* format the column appropriately (either text or HOURmin) */
  if (column == 0)
    {
      /* erase the entire row once before we draw the first column. This reduces screen flicker, and is more efficient
       * than erasing each individual cell before re-filling it. */
      rowRect.topLeft.x = bounds->topLeft.x;
      rowRect.topLeft.y = bounds->topLeft.y;
      WinEraseRectangle(&rowRect, 0);
      /* draw the hour in the large font */
      prevFont = FntSetFont(boldFont);
      if (idx & 0x0F)
        {
          if ((idx & 0x0F) > 9)
            {
              txt[0] = '1';     /* actually idx is hour/mins */
              txt[1] = 48 + (idx & 0x0F) - 10;
              txt[2] = NULL;
              WinDrawChars(txt, 2, bounds->topLeft.x, bounds->topLeft.y);
            }
          else
            {
              txt[0] = 48 + (idx & 0x0F);       /* actually idx is hour/mins */
              txt[1] = NULL;
              WinDrawChars(txt, 1, bounds->topLeft.x + 6, bounds->topLeft.y);
            }
        }
      /* draw the minutes in the normal font */
      txt[0] = 48 + ((idx & 0xE0) >> 5);
      txt[1] = 48 + (5 * ((idx & 0x10) >> 4));
      txt[2] = NULL;
      FntSetFont(stdFont);
      WinDrawChars(txt, 2, bounds->topLeft.x + 12, bounds->topLeft.y);
    }
  else
    {
      /* has this row been marked as chargeable, by underlining the current cell */
      if (idx < 0)
        {
          /* fixup index */
          idx = (-idx) - 1;
          /* turn on underlining */
          WinSetUnderlineMode(solidUnderline);
        }
      listHand = DmQueryRecord(TSDatabase, column);
      listPtr = MemHandleLock(listHand);

      /* use the resource translation table to convert index to actual entry index */
      idx = listPtr->transTable[idx];

      /* get the string to put in the cell */
      StrCopy(txt, listPtr->cats[idx]);
      MemHandleUnlock(listHand);

      /* trim the client string to fit within the cell */
      prevFont = FntSetFont(stdFont);

      /* draw the cell contents */
      txtWidth = bounds->extent.x;
      txtLen = StrLen( txt );
      FntCharsInWidth( txt, &txtWidth, &txtLen, &txtFitsInWidth );
      txt[ txtLen ] = NULL;
      WinDrawChars( txt, txtLen, bounds->topLeft.x, bounds->topLeft.y );

      /* turn off underlining */
      WinSetUnderlineMode(noUnderline);
    }
  /* restore the previous font */
  FntSetFont(prevFont);

  /* end fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  e_Delete(txt);
  CALLBACK_EPILOGUE;
  
}

/******************************************************************
 * Tally up the next entry record in the main table
 ******************************************************************/
void MainFormTableRowTally(UInt recIdx,
                                  TSDurationRecType* totalDuration,
                                  TSDurationRecType* totalChargeableDuration )
{
  VoidHand recHnd;
  TSEntryRecType* recPtr;
  UInt recAttr;
  Int itemInt;

  /* get the attributes and the entry record this row corresponds to */
  DmRecordInfo(TSDatabase, TSPresentDayRecIdx + recIdx, &recAttr, NULL, NULL);
  recHnd = DmQueryRecord(TSDatabase, TSPresentDayRecIdx + recIdx);
  recPtr = MemHandleLock(recHnd);
  /* tally up hours and minutes for daily total */
  itemInt = recPtr->hours;
  totalDuration->mins += ((itemInt & 0xF0) >> 4);
  totalDuration->hours += (itemInt & 0x0F);
  if (totalDuration->mins >= 12)
    {
      totalDuration->mins -= 12;
      totalDuration->hours += 1;
    }
  /* chargeable entry, so tally up hours and minutes for chargeable daily total */
  if (recAttr & 0x0008)
    {
      totalChargeableDuration->mins += ((itemInt & 0xF0) >> 4);
      totalChargeableDuration->hours += (itemInt & 0x0F);
      if (totalChargeableDuration->mins >= 12)
        {
          totalChargeableDuration->mins -= 12;
          totalChargeableDuration->hours += 1;
        }
    }
  MemHandleUnlock(recHnd);
}

/******************************************************************
 * Fill the main form with entries for the present day (if any).
 ******************************************************************/
void
MainFormTableDraw(void)
{
  Boolean dayHasEntries;
  Byte numEntries;
  UInt recIdx;
  UInt recAttr;
  UInt itemInt;
  Word col;
  Word row;
  VoidHand recHnd;
  VoidPtr recPtr;
  FormPtr formPtr;
  ControlPtr controlPtr;
///  Char txt[3];
  Char * txt;
  Byte txtLen;
  TSDurationRecType totalDuration;
  TSDurationRecType totalChargeableDuration;
  Coord jox;
  Coord joy; // my little hack. 

  txt = e_New(3);

  /* get a pointer to the main form as it may not be the visible one at the moment */
  formPtr = FrmGetFormPtr(formID_tsMain);
  /* get the table pointer */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, tableID_tsMain));
  /* reset totals */
  totalDuration.hours = totalDuration.mins = 0;
  totalChargeableDuration.hours = totalChargeableDuration.mins = 0;
  /* find the number of entries in the current day */
  if (TSPresentDayRecExists)
    {
      /* get the number of entries from the day record */
      recHnd = DmQueryRecord(TSDatabase, TSPresentDayRecIdx);
      recPtr = MemHandleLock(recHnd);
      numEntries = ((TSDayRecType *) recPtr)->numEntries;
      MemHandleUnlock(recHnd);
      /* go through all the entries prior to the first record that actually appears in the table, building totals */
      for (recIdx = 1; recIdx < TSTableTopEntry + 1; recIdx++)
        {
          MainFormTableRowTally(recIdx, &totalDuration, &totalChargeableDuration );
        }
    }
  else
    {
      /* no entries for today */
      numEntries = 0;
      recIdx = 0;
    }
  dayHasEntries = numEntries > 0;
  /* go through the remaing records, building the rows in the table */
  for (row = 0; row < TSMainFormTableRows; row++, recIdx++)
    {
      if (recIdx > 0 && recIdx <= numEntries)
        {
          MainFormTableRowTally(recIdx, &totalDuration, &totalChargeableDuration );
          /* get the attributes of the entry record this row corresponds to */
          DmRecordInfo(TSDatabase, TSPresentDayRecIdx + recIdx, &recAttr, NULL, NULL);
          /* get the entry record this row corresponds to */
          recHnd = DmQueryRecord(TSDatabase, TSPresentDayRecIdx + recIdx);
          recPtr = MemHandleLock(recHnd);
          /* fill the row items with record data */
          /* set hour/minute/chargeable column */
          TblSetItemStyle((TablePtr) controlPtr, row, 0, customTableItem);
          TblSetItemInt((TablePtr) controlPtr, row, 0, ((TSEntryRecType *) recPtr)->hours);

          /* set client column */
          TblSetItemStyle((TablePtr) controlPtr, row, 1, customTableItem);
          /* translate the client entry index into the actual client index using the client trans table */
          if ((recAttr & 0x0008) && (TSAppPrefs.prefFlags & TSPrefUlineClientFlag))
            {
              /* entry is chargeable, -ve client index indicates chargeable rows */
              itemInt = -(((TSEntryRecType *) recPtr)->clientIdx + 1);  /* have to add one as idx 0 = 'none', so how do we make
                                                                         * -0 indicate underlining? */
            }
          else
            {
              /* entry is not chargeable, +ve client index indicates non-chargeable rows */
              itemInt = ((TSEntryRecType *) recPtr)->clientIdx;
            }
          TblSetItemInt((TablePtr) controlPtr, row, 1, itemInt);

          /* set project column */
          TblSetItemStyle((TablePtr) controlPtr, row, 2, customTableItem);
          /* translate the client entry index into the actual client index using the client trans table */
          TblSetItemInt((TablePtr) controlPtr, row, 2, ((TSEntryRecType *) recPtr)->projectIdx);

          /* set task column */
          TblSetItemStyle((TablePtr) controlPtr, row, 3, customTableItem);
          /* translate the client entry index into the actual client index using the client trans table */
          TblSetItemInt((TablePtr) controlPtr, row, 3, ((TSEntryRecType *) recPtr)->taskIdx);

          /* finished setting up columns */
          MemHandleUnlock(recHnd);
          /* data for this row, make it usable and selectable */
          TblSetRowUsable((TablePtr) controlPtr, row, true);
          /* mark the row invalid, so it re-draws */
          TblMarkRowInvalid((TablePtr) controlPtr, row);
        }
      else
        {
          /* no data for this row, make it unusable */
          TblSetRowUsable((TablePtr) controlPtr, row, false);
        }
    }
  /* total the remaining entries (if any) */
  for (; recIdx <= numEntries; recIdx++)
    {
      MainFormTableRowTally(recIdx, &totalDuration, &totalChargeableDuration );
    }
  /* make all columns in the table usable */
  for (col = 0; col < TSMainFormTableCols; col += 1)
    {
      TblSetColumnUsable((TablePtr) controlPtr, col, true);
    }
  /* set column spacing */
  TblSetColumnSpacing((TablePtr) controlPtr, 0, 3);
  TblSetColumnSpacing((TablePtr) controlPtr, 1, 3);
  TblSetColumnSpacing((TablePtr) controlPtr, 2, 3);
  TblSetColumnSpacing((TablePtr) controlPtr, 3, 0);
  /* set the custom drawn columns */
  TblSetCustomDrawProcedure((TablePtr) controlPtr, 0, (TableDrawItemFuncPtr) MainFormTableDrawCell);   /* hour / mins column */
  TblSetCustomDrawProcedure((TablePtr) controlPtr, 1, (TableDrawItemFuncPtr)MainFormTableDrawCell);   /* client column */
  TblSetCustomDrawProcedure((TablePtr) controlPtr, 2, (TableDrawItemFuncPtr)MainFormTableDrawCell);   /* project column */
  TblSetCustomDrawProcedure((TablePtr) controlPtr, 3, (TableDrawItemFuncPtr)MainFormTableDrawCell);   /* task column */
  /* redraw the table */
  TblDrawTable((TablePtr) controlPtr);
  if (dayHasEntries)
    {
      /* draw a little line to indicate it's a total */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, labelID_tsTotal));

/// [jo] the following is VERBOTEN:
//      col = controlPtr->bounds.topLeft.x;
//      row = controlPtr->bounds.topLeft.y;
FrmGetObjectPosition(formPtr, FrmGetObjectIndex(formPtr, labelID_tsTotal), &jox, &joy);
  // [jo] above fix should return relevant values:
  // then say: (clumsy)
     col = jox;
     row = joy; 

      WinDrawLine(col, row - 1, col + 22, row - 1);

      /* draw the hour and minute totals if req'd */
      if (totalDuration.hours > 0)
        {
          if (totalDuration.hours < 10)
            {
              /* hopefully less code than calling StrIToA? */
              txt[0] = 48 + totalDuration.hours;
              txt[1] = NULL;
              txtLen = 1;
              col += 6;
            }
          else
            {
              StrIToA(txt, totalDuration.hours);
              txtLen = StrLen(txt);
            }
          FntSetFont(boldFont);
          WinDrawChars(txt, txtLen, col, row);
        }
      /* draw the minutes in the normal font */
      FntSetFont(stdFont);
      txt[0] = 48 + ((totalDuration.mins & 0x0E) >> 1);
      txt[1] = 48 + (5 * (totalDuration.mins & 0x01));
      txt[2] = NULL;
///      WinDrawChars(txt, 2, controlPtr->bounds.topLeft.x + 12, row);
      WinDrawChars(txt, 2, jox + 12, row);

      /* Draw the chargeable hour and minute totals if req'd.
       * Note, this preference is 'inverted', 1 = Off, 0 = On so it's on by default */
      if (!(TSAppPrefs.prefFlags & TSPrefChargeTotalsFlag))
        {
///          col = controlPtr->bounds.topLeft.x + 32;
///          row = controlPtr->bounds.topLeft.y;
          col = jox + 32;
          row = joy;

          WinDrawLine(col, row - 1, col + 22, row - 1);
          if (totalChargeableDuration.hours > 0)
            {
              if (totalChargeableDuration.hours < 10)
                {
                  /* hopefully less code than calling StrIToA? */
                  txt[0] = 48 + totalChargeableDuration.hours;
                  txt[1] = NULL;
                  txtLen = 1;
                  col += 6;
                }
              else
                {
                  StrIToA(txt, totalChargeableDuration.hours);
                  txtLen = StrLen(txt);
                }
              FntSetFont(boldFont);
              WinDrawChars(txt, txtLen, col, row);
            }
          /* draw the minutes in the normal font */
          FntSetFont(stdFont);
          txt[0] = 48 + ((totalChargeableDuration.mins & 0x0E) >> 1);
          txt[1] = 48 + (5 * (totalChargeableDuration.mins & 0x01));
          txt[2] = NULL;
///          WinDrawChars(txt, 2, controlPtr->bounds.topLeft.x + 32 + 12, row);
          WinDrawChars(txt, 2, jox + 32 + 12, row);
       }
    }
  /* Finally update the on-screen scroll buttons */
  MainFormScrollButtonUpdate(formPtr);  
  e_Delete(txt);
}

/******************************************************************
 * Scroll the table in the specified direction (pageUpChr or pageDownChr)
 ******************************************************************/
void
MainFormTableScroll(Byte dir)
{
  Word numEntries;
  VoidHand recHand;
  TSDayRecType *recPtr;

  /* Find the number of entries in the current day (if any) */
  if (TSPresentDayRecExists)
    {
      recHand = DmQueryRecord(TSDatabase, TSPresentDayRecIdx);
      recPtr = MemHandleLock(recHand);
      numEntries = recPtr->numEntries;
      MemHandleUnlock(recHand);
    }
  else
    {
      numEntries = 0;
    }
  TableScroll(&TSTableTopEntry, TSMainFormTableRows, numEntries, dir);
}

/******************************************************************
 * Update the on-screen scroll up/down buttons.
 ******************************************************************/
void
MainFormScrollButtonUpdate(FormPtr formPtr)
{
  Byte numEntries;
  VoidHand recHand;
  TSDayRecType *recPtr;

  /* Find the number of entries in the current day (if any) */
  if (TSPresentDayRecExists)
    {
      recHand = DmQueryRecord(TSDatabase, TSPresentDayRecIdx);
      recPtr = MemHandleLock(recHand);
      numEntries = recPtr->numEntries;
      MemHandleUnlock(recHand);
    }
  else
    {
      numEntries = 0;
    }
  /* Update the scroll buttons */
  ScrollButtonUpdate(formPtr, TSTableTopEntry, TSMainFormTableRows, numEntries);
}

/*** End of section! ***/

// ============================================================================ //
// DetailForm:
// #include "DetailForm.c"

/******************************************************************
 * Add the timed time to the detail form entry time
 ******************************************************************/
void DetailFormAddTime(long deltaSecs)
{
  Byte entryHours;
  Byte entryMins;
  Long deltaHours;
  Long deltaMins;
  Long deltaFiveMins;

  /* Convert the seconds time to hours and minutes */
  deltaHours = deltaSecs / 3600;
  deltaMins = ( deltaSecs - (deltaHours * 3600) ) / 60;
  deltaFiveMins = deltaMins / 5;

  /* Check we have sufficient time to add */
  if (deltaHours == 0 && deltaFiveMins == 0)
    {
      /* Not enough time has elapsed to add, alert the user */
      FrmAlert(alertID_tsTimerTooShort);
    }
  else
    {
      /* Sufficient time has elapsed to add */

      /* Extract current hours / minutes from details form */
      DetailFormGetDuration(FrmGetActiveForm(), &entryHours, &entryMins);

      /* Add delta hours to entry hours */
      entryMins += deltaFiveMins;
      if (entryMins >= 12)
        {
          entryMins -= 12;
          entryHours ++;
        }

      /* Check if we've exceeded time limit for single entry */
      if (entryHours + deltaHours > 15)
      {
        /* Exceeded maximum representable time in current database format, alert user */
        FrmAlert(alertID_tsTimerTooLong);
        entryHours = 15;
        entryMins = 11;
      }
      else
      {
        entryHours += deltaHours;
      }

      /* Save current hours / minutes back to details form */
      DetailFormSetDuration(FrmGetActiveForm(), entryHours, entryMins);
    }
}

/******************************************************************
 * Close the detail form, releasing any dynamic memory used.
 ******************************************************************/

void DetailFormClose(FormPtr formPtr)
{
// [$jvs] [jo] WE MUST STILL FIX THIS AS LEAVES UNFREED MEMORY!

  UInt id;
  ListPtr listPtr;
  FieldPtr fieldPtr;
  VoidHand hand;

  /* free dynamically allocated field memory */
  fieldPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsDesc));
  hand = (VoidHand) FldGetTextHandle(fieldPtr);
  if (hand)
    {
      /* set handle to null so form closing doesn't cause problems */
      FldSetTextHandle(fieldPtr, 0);
      /* release the dynamically allocated memory */
      MemHandleFree(hand);
    }

 /* free the memory used by popup list text ptr arrays */
  for (id = listID_tsClient; id <= listID_tsTask; id++)
    {
      listPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, id));
      MemPtrFree(listPtr->itemsText);
//      listPtr->itemsText = NULL; /// [jo] 12-8-2005 this is causing trouble. 
      /// we are told to make the appropriate form manager calls:
      /// 14-8-2005 seems ok if simply don't set value to null!
      /// still an ugly hack, may break in the future, again.
    }
// the correct fix for the above is probably to simply make a linked list, erasing
// it when leaving!
}




/******************************************************************
 * Delete the currently selected entry record from the timesheet
 * database. May also delete the current day (if the last entry for
 * the day is deleted). Returns the number of entries LEFT in the
 * day after the deletion (0 if day has been deleted).
 ******************************************************************/
UInt
DetailFormDeleteEntry(UInt entryRecIdx)
{
  Byte numEntries;
  Byte entryNum;
  UInt idx;
  VoidHand recHand;
  TSDayRecType *dayPtr;
  TSEntryRecType *entryPtr;

  /* delete the current entry from the database */
  DmRemoveRecord(TSDatabase, entryRecIdx);
  /* deleted an entry from the present day */
  recHand = DmQueryRecord(TSDatabase, TSPresentDayRecIdx);
  dayPtr = MemHandleLock(recHand);
  numEntries = (dayPtr->numEntries) - 1;
  /* any entries left for present day? */
  if (numEntries == 0)
    {
      /* no, delete day record... */
      MemHandleUnlock(recHand);
      DmRemoveRecord(TSDatabase, TSPresentDayRecIdx);
      /* and update globals accordingly */
      TSPresentDayRecExists = false;
    }
  else
    {
      /* yes, leave day record intact (but save the new # entries) */
      DmWrite(dayPtr, 0, &numEntries, sizeof(Byte));
      MemHandleUnlock(recHand);
      /* fix all the entries after the deleted entry by reducing their entryNum by one */
      for (idx = entryRecIdx; idx <= TSPresentDayRecIdx + numEntries; idx++)
        {
          recHand = DmGetRecord(TSDatabase, idx);
          entryPtr = MemHandleLock(recHand);
          entryNum = entryPtr->entryNum;
          entryNum--;
          DmWrite(entryPtr, (VoidPtr) (&(entryPtr->entryNum)) - (VoidPtr) entryPtr, &entryNum, sizeof(Byte));
          MemHandleUnlock(recHand);
          DmReleaseRecord(TSDatabase, idx, true);
        }
    }
  /* Check if we've deleted the timer entry record (if any) */
  if (TSEntryTimer.recIdx > 0)
  {
    if (TSEntryTimer.recIdx == entryRecIdx)
    {
      /* Stop the timer as the related record has gone */
      TSEntryTimer.recIdx = 0;
      TSEntryTimer.secs = 0;
    }
    else if (TSEntryTimer.recIdx > entryRecIdx)
    {
      /* Adjust the timer record down to account for deleted record */
      if (numEntries == 0)
      {
        /* Deleted entry AND day record that contained entry */
        TSEntryTimer.recIdx -= 2;
      }
      else
      {
        /* Just deleted a single entry */
        TSEntryTimer.recIdx -= 1;
      }
    }
  }
  return numEntries;
}

/******************************************************************
 * Event handler for the Details form.
 ******************************************************************/
Boolean
DetailFormEventHandler(EventPtr event)
{
  ControlPtr controlPtr;
  FormPtr formPtr;
  RectangleType rect;
  Boolean handled = false;
  Coord jox;
  Coord joy; // my little hack. 

  if (event->eType == fldChangedEvent)
    {
      /* desc field has changed, update scroll buttons */
      DetailFormScrollBarUpdate(FrmGetActiveForm());
      handled = true;
    }
  else if (event->eType == sclRepeatEvent)
    {
      /* scroll the field */
      DetailFormScroll(FrmGetActiveForm(), event->data.sclRepeat.newValue - event->data.sclRepeat.value);
    }
  else if (event->eType == ctlSelectEvent)
    {
      /* a control button was pressed and released */
      if (event->data.ctlSelect.controlID == buttonID_tsOk)
        {
          /* add a new entry or modify an existing one */
          if (TSEntryTimer.secs > 0 && TSEntryTimer.recIdx == TSEditEntryIdx)
            {
              /* We have a timer running, does the user want to stop it or keep timing? */
              if ((TSAppPrefs.prefFlags & TSPrefAlwaysTimeFlag)
                  || FrmAlert(alertID_tsKeepTiming) == 0)
              {
                /* Do nothing, keep timing */
              }
              else
              {
                /* Stop entry timer and update form with time elapsed prior to entry update */
                DetailFormAddTime((TimGetSeconds() - TSEntryTimer.secs) + 1);
                TSEntryTimer.secs = 0;
                TSEntryTimer.recIdx = 0;
              }
            }
          if (TSEditEntryIdx == 0)
            {
              /* add a new entry, updating timing with the new entry index */
              if (TSEntryTimer.secs > 0 && TSEntryTimer.recIdx == TSEditEntryIdx)
              {
                TSEntryTimer.recIdx = DetailFormNewEntry();
              }
              else
              {
                DetailFormNewEntry();
              }
            }
          else
            {
              /* modify the existing, selected entry */
              DetailFormToEntryRec(TSEditEntryIdx);
            }
          /* show the main form again */
          DetailFormClose(FrmGetActiveForm());
          FrmReturnToForm(formID_tsMain);
          MainFormRefresh();
          handled = true;
        }
      else if (event->data.ctlSelect.controlID == buttonID_tsCancel)
        {
          if (TSEntryTimer.secs > 0 && TSEntryTimer.recIdx == TSEditEntryIdx)
          {
            /* Stop timing the current entry */
            TSEntryTimer.secs = 0;
            TSEntryTimer.recIdx = 0;
          }
          /* cancel details form and show the main form again */
          DetailFormClose(FrmGetActiveForm());
          FrmReturnToForm(formID_tsMain);
          MainFormRefresh();
          handled = true;
        }
      else if (event->data.ctlSelect.controlID == buttonID_tsDelete)
        {
          /* delete the current entry record (and possibly the current day) */
          /* first confirm deletion */
          if (FrmAlert(alertID_tsConfirmDel) == 0)
            {
              if (TSEntryTimer.secs > 0 && TSEntryTimer.recIdx == TSEditEntryIdx)
              {
               /* Stop timing the current entry */
               TSEntryTimer.secs = 0;
               TSEntryTimer.recIdx = 0;
              }
              /* delete the current entry */
              DetailFormClose(FrmGetActiveForm());
              FrmReturnToForm(formID_tsMain);
              DetailFormDeleteEntry(TSEditEntryIdx);
              /* adjust the table top entry as we've just deleted record */
              if (TSTableTopEntry > 0)
                {
                  TSTableTopEntry--;
                }
              MainFormRefresh();
            }                   /* else do nothing, deletion wasn't confirmed */
          handled = true;
        }
      else if (event->data.ctlSelect.controlID == buttonID_tsStartStop)
        {
          if (TSEntryTimer.secs == 0)
            {
              /* Start entry timer */
              TSEntryTimer.secs = TimGetSeconds();
              TSEntryTimer.recIdx = TSEditEntryIdx;
            }
          else if (TSEntryTimer.recIdx == TSEditEntryIdx)
            {
              /* Stop entry timer */
              formPtr = FrmGetActiveForm();
              controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, bitmapID_tsStopWatch));
              DetailFormDrawStopWatch(formPtr, controlPtr);
              DetailFormAddTime((TimGetSeconds() - TSEntryTimer.secs) + 1);
              TSEntryTimer.secs = 0;
              TSEntryTimer.recIdx = 0;
            }
          else
            {
              /* There's currently a timer running on another event */
              FrmAlert(alertID_tsTimerOnOtherEntry);
            }
          handled = true;
        }
    }
  else if (event->eType == popSelectEvent)
    {
      if (event->data.popSelect.listID == listID_tsClient)
        {
          /* entry in client list has been selected */
          if (event->data.popSelect.selection == (TSAppPrefs.numCatEntries[0] - 1))
            {
              /* last entry (the 'Edit...' entry) has been selected */
              formPtr = FrmGetActiveForm();
              controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsClient));
              LstSetSelection((ListPtr) controlPtr, event->data.popSelect.priorSelection);
              /* show the edit form */
              TSEditCatRecIdx = 1;
              FrmPopupForm(formID_tsEditCat);
            }
        }
      else if (event->data.popSelect.listID == listID_tsProject)
        {
          /* entry in client list has been selected */
          if (event->data.popSelect.selection == (TSAppPrefs.numCatEntries[1] - 1))
            {
              /* last entry (the 'Edit...' entry) has been selected */
              formPtr = FrmGetActiveForm();
              controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsProject));
              LstSetSelection((ListPtr) controlPtr, event->data.popSelect.priorSelection);
              /* show the edit form */
              TSEditCatRecIdx = 2;
              FrmPopupForm(formID_tsEditCat);
              handled = true;
            }
        }
      else if (event->data.popSelect.listID == listID_tsTask)
        {
          /* entry in client list has been selected */
          if (event->data.popSelect.selection == (TSAppPrefs.numCatEntries[2] - 1))
            {
              /* last entry (the 'Edit...' entry) has been selected */
              formPtr = FrmGetActiveForm();
              controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsTask));
              LstSetSelection((ListPtr) controlPtr, event->data.popSelect.priorSelection);
              /* show the edit form */
              TSEditCatRecIdx = 3;
              FrmPopupForm(formID_tsEditCat);
              handled = true;
            }
        }
    }
  else if (event->eType == menuEvent)
    {
      /* a menu item has been selected (only menu options presently are Undo/Cut/Copy/Paste) */
      formPtr = FrmGetActiveForm();
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsDesc));
      if (event->data.menu.itemID == menuitemID_tsUndo)
        {
          FldUndo((FieldPtr) controlPtr);
        }
      else if (event->data.menu.itemID == menuitemID_tsCut)
        {
          FldCut((FieldPtr) controlPtr);
        }
      else if (event->data.menu.itemID == menuitemID_tsCopy)
        {
          FldCopy((FieldPtr) controlPtr);
        }
      else if (event->data.menu.itemID == menuitemID_tsPaste)
        {
          FldPaste((FieldPtr) controlPtr);
        }
      handled = true;
    }
  else if (event->eType == nilEvent)
    {
      /* If we're timing, toggle the stopwatch on/off once per second */
      if (TSEntryTimer.secs > 0 && TSEntryTimer.recIdx == TSEditEntryIdx)
        {
          formPtr = FrmGetActiveForm();
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, bitmapID_tsStopWatch));
          if (TimGetSeconds() & 0x01)
            {
              DetailFormDrawStopWatch(formPtr, controlPtr);
            }
          else
            {
              /* Erase the bitmap */
  FrmGetObjectPosition(formPtr, FrmGetObjectIndex(formPtr, bitmapID_tsStopWatch), &jox, &joy);
              rect.topLeft.x = jox;
              rect.topLeft.y = joy;
              rect.extent.x = 7;
              rect.extent.y = 8;
              WinEraseRectangle(&rect, 0);
            }
        }
    }
  else if (event->eType == frmOpenEvent)
    {
      /* the Detail form has been opened */
      formPtr = FrmGetActiveForm();
      DetailFormInit(formPtr);
      FrmDrawForm(formPtr);
      FormUpdateDay(&TSPresentDayDate);
      FrmSetFocus(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsDesc));
      handled = true;
    }

  return handled;
}

/******************************************************************
 * Utility function to save a little repeated code. Draw the stopwatch bitmap
 ******************************************************************/
void DetailFormDrawStopWatch(FormPtr formPtr, ControlPtr controlPtr)
{ /// [jo] (can be made smaller)
  BitmapPtr bitmapPtr;
  VoidHand hand;
  Coord jox;
  Coord joy; // my little hack. 

  /* Draw the bitmap */
  hand = DmGet1Resource('Tbmp', bitmapID_tsStopWatch);
  bitmapPtr = MemHandleLock(hand);
  FrmGetObjectPosition(formPtr, FrmGetObjectIndex(formPtr, bitmapID_tsStopWatch), &jox, &joy);
  WinDrawBitmap(bitmapPtr, jox, joy);
  MemHandleUnlock(hand);
  DmReleaseResource(hand);
}

/******************************************************************
 * Initialise the details form, which is about to be shown.
 ******************************************************************/
void
DetailFormInit(FormPtr formPtr)
{
  UInt recAttr;
  Byte clientIdx;
  Byte projectIdx;
  Byte taskIdx;
  Byte hourIdx;
  Byte minuteIdx;
  TSEntryRecType *entryPtr;
  VoidHand entryHand;
  VoidHand hand;
  VoidHand catHand;
  Ptr ptr;
  TSCatRecType *catPtr;
  CharPtr str;
  ControlPtr controlPtr;
  ListPtr clientListPtr;
  ListPtr projectListPtr;
  ListPtr taskListPtr;
  FieldAttrType attr;

  /* set the client list for the form */
  clientListPtr = DetailFormInitPopupList(listID_tsClient, listpopupID_tsClient, 1, TSAppPrefs.numCatEntries[0]);
  /* set the project list for the form */
  projectListPtr = DetailFormInitPopupList(listID_tsProject, listpopupID_tsProject, 2, TSAppPrefs.numCatEntries[1]);
  /* set the task list for the form */
  taskListPtr = DetailFormInitPopupList(listID_tsTask, listpopupID_tsTask, 3, TSAppPrefs.numCatEntries[2]);
  /* editing new or existing entry? */
  if (TSEditEntryIdx == 0)
    {
      /* we are editing a potentially *new* entry in the present day */
      /* make the delete button invisible */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsDelete));
      CtlSetUsable(controlPtr, false);
      /* set the client/project/task indexes depending on the Auto Categories preference */
      if (TSAppPrefs.prefFlags & TSPrefAutoCatsFlag)
        {
          clientIdx = TSAppPrefs.newCatEntryIdx[0];
          projectIdx = TSAppPrefs.newCatEntryIdx[1];
          taskIdx = TSAppPrefs.newCatEntryIdx[2];
        }
      else
        {
          clientIdx = 0;
          projectIdx = 0;
          taskIdx = 0;
        }
      /* set the hours and minutes depending on the Auto Durations preference */
      if (TSAppPrefs.prefFlags & TSPrefAutoDurFlag)
        {
          hourIdx = (TSAppPrefs.newCatHours) & 0x0F;    /* hours occupy low 4 bits */
          minuteIdx = (TSAppPrefs.newCatHours) >> 4;    /* minutes occupy high 4 bits */
        }
      else
        {
          hourIdx = 0;
          minuteIdx = 0;
        }
      /* set the checkbox on/off depending on the global app prefs */
      CtlSetValue(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsCharge)),
                  (TSAppPrefs.prefFlags & TSPrefDefaultChargeFlag));
      /* make the field blank */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsDesc));
      hand = MemHandleNew(sizeof(Char));
      /// [jo] MUCH later will need to free this. fix me!
      ptr = MemHandleLock(hand);
      MemSet(ptr, 1, 0);        /* make sure the field is blank */
      MemHandleUnlock(hand);
      FldSetTextHandle((FieldPtr) controlPtr, (Handle) hand);
    }
  else
    {
      /* we are editing an exiting entry in the present day */
      entryHand = DmQueryRecord(TSDatabase, TSEditEntryIdx);
      entryPtr = MemHandleLock(entryHand);
      /* set the client list index */
      clientIdx = entryPtr->clientIdx;
      /* set the project list index */
      projectIdx = entryPtr->projectIdx;
      /* set the task list index */
      taskIdx = entryPtr->taskIdx;
      /* set the hours and minutes */
      hourIdx = (entryPtr->hours) & 0x0F;       /* hours occupy low 4 bits */
      minuteIdx = (entryPtr->hours) >> 4;       /* minutes occupy high 4 bits */
      /* set the checkbox on/off depending on entry chargeable bit */
      DmRecordInfo(TSDatabase, TSEditEntryIdx, &recAttr, NULL, NULL);
      CtlSetValue(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsCharge)), (recAttr & TSEntryChargedAttr) >> 3);
      /* set the desc field using a dynamic handle */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsDesc));
      str = (CharPtr) (((VoidPtr) entryPtr) + sizeof(TSEntryRecType));  /* desc starts after structure in memory */
      hand = MemHandleNew(sizeof(Char) * (StrLen(str) + 1));
      /// [jo] MUCH later will need to free this. fix me!
      /* copy into dynamic mem the existing string */
      ptr = MemHandleLock(hand);
      StrCopy((CharPtr) ptr, str);
      MemHandleUnlock(hand);
      FldSetTextHandle((FieldPtr) controlPtr, (Handle) hand);
      /* finished setting up the form for existing entry */
      MemHandleUnlock(entryHand);
    }
  /* have the field send event to maintain the scroll bar. */
  FldGetAttributes((FieldPtr) controlPtr, &attr);
  attr.hasScrollBar = true;
  FldSetAttributes((FieldPtr) controlPtr, &attr);
  /* set the client list */
  catHand = DmQueryRecord(TSDatabase, 1);
  catPtr = MemHandleLock(catHand);
  clientIdx = catPtr->transTable[clientIdx];
  MemHandleUnlock(catHand);
  LstSetSelection(clientListPtr, clientIdx);
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listpopupID_tsClient));
  CtlSetLabel(controlPtr, LstGetSelectionText(clientListPtr, clientIdx));
  /* set the project list */
  catHand = DmQueryRecord(TSDatabase, 2);
  catPtr = MemHandleLock(catHand);
  projectIdx = catPtr->transTable[projectIdx];
  MemHandleUnlock(catHand);
  LstSetSelection(projectListPtr, projectIdx);
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listpopupID_tsProject));
  CtlSetLabel(controlPtr, LstGetSelectionText(projectListPtr, projectIdx));
  /* set the task list */
  catHand = DmQueryRecord(TSDatabase, 3);
  catPtr = MemHandleLock(catHand);
  taskIdx = catPtr->transTable[taskIdx];
  MemHandleUnlock(catHand);
  LstSetSelection(taskListPtr, taskIdx);
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listpopupID_tsTask));
  CtlSetLabel(controlPtr, LstGetSelectionText(taskListPtr, taskIdx));
  /* set the hours and minutes */
  DetailFormSetDuration(formPtr, hourIdx, minuteIdx);
  /* initialise the scroll bar */
  DetailFormScrollBarUpdate(formPtr);
}

/******************************************************************
 * A new entry has been added at the specified category index.
 ******************************************************************/
void
DetailFormPopupListAdd(UInt listID, UInt popupID, UInt catRecIdx, UInt numCatEntries, UInt addedCatIdx)
{
  Word selected;
  FormPtr formPtr;
  ListPtr listPtr;

  /* get the list's current selection */
  formPtr = FrmGetFormPtr(formID_tsDetail);
  listPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID));
  selected = LstGetSelection(listPtr);
  /* re-fill the list (including the added entry) */
  DetailFormFillPopupList(listID, popupID, catRecIdx, numCatEntries, true);
  /* reselect the appropriate selection */
  if (selected >= addedCatIdx)
    {
      selected += 1;
    }
  LstSetSelection(listPtr, selected);
}

/******************************************************************
 * An existing entry has been removed at the specified category index.
 ******************************************************************/
void
DetailFormPopupListDelete(UInt listID, UInt popupID, UInt catRecIdx, UInt numCatEntries, UInt deletedCatIdx)
{
  Word selected;
  FormPtr formPtr;
  ListPtr listPtr;

  /* get the list's current selection */
  formPtr = FrmGetFormPtr(formID_tsDetail);
  listPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID));
  selected = LstGetSelection(listPtr);
  /* re-fill the list (including the added entry) */
  DetailFormFillPopupList(listID, popupID, catRecIdx, numCatEntries, true);
  /* reselect the appropriate selection */
  if (selected == deletedCatIdx)
    {
      selected = 0;
    }
  else if (selected > deletedCatIdx)
    {
      selected -= 1;
    }
  LstSetSelection(listPtr, selected);
}

/******************************************************************
 * An existing entry has been renamed at the specified category index.
 ******************************************************************/
void
DetailFormPopupListRename(UInt listID, UInt popupID, UInt catRecIdx, UInt numCatEntries,
                          UInt renamedCatIdx, UInt renameToCatIdx)
{
  Word selected;
  FormPtr formPtr;
  ListPtr listPtr;

  /* get the list's current selection */
  formPtr = FrmGetFormPtr(formID_tsDetail);
  listPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID));
  selected = LstGetSelection(listPtr);
  /* re-fill the list (including the added entry) */
  DetailFormFillPopupList(listID, popupID, catRecIdx, numCatEntries, true);
  /* reselect the appropriate selection */
  if (renamedCatIdx == renameToCatIdx)
    {
      /* do nothing, renamed category hasn't changed place */
      return;
    }
  else if (selected == renamedCatIdx)
    {
      /* change selection to new position of renamed cat */
      selected = renameToCatIdx;
    }
  LstSetSelection(listPtr, selected);
}

/******************************************************************
 * Fill the supplied popup list with entries from the specified category.
 * If freeExisting is true, the memory used by existing list entries if released first.
 ******************************************************************/
ListPtr
DetailFormFillPopupList(UInt listID, UInt popupID, UInt catRecIdx, UInt numCatEntries,
                        Boolean freeExisting)
{
  VoidHand catHand;
  TSCatRecType *catPtr;
  CharPtr *catStrPtr;
  ListPtr listPtr;
  FormPtr formPtr;

  /* get the list control */
  formPtr = FrmGetFormPtr(formID_tsDetail);
  listPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID));
  /* open the category record */
  catHand = DmGetRecord(TSDatabase, catRecIdx);
  catPtr = MemHandleLock(catHand);
  /* free memory for old pointer array if it exists */
  if (freeExisting)
    {
      MemPtrFree(listPtr->itemsText);
      /// listPtr->itemsText = NULL;
      /// [jo] this is a stuffup. Try simply removing NULL allocation ??? 13-8-2005.
      /// yep, seems to work!
    }
  /* rebuild the pointer array in case OS has moved records around in memory */
  catStrPtr = CatBuildPtrArray(catPtr, numCatEntries);
  /* fill it with the supplied entries */
  LstSetListChoices(listPtr, catStrPtr, numCatEntries);
  /* set the list length depending on the number of entries so we don't show funny looking half-empty lists */
  if (numCatEntries < 10)
    {
      LstSetHeight(listPtr, numCatEntries);
    }
  else
    {
      LstSetHeight(listPtr, 10);
    }
  /* release the category record */
  MemHandleUnlock(catHand);
  DmReleaseRecord(TSDatabase, catRecIdx, true);
  return listPtr;
}

/******************************************************************
 * Set up the details control for the supplied list.
 ******************************************************************/
ListPtr
DetailFormInitPopupList(UInt listID, UInt popupID, UInt catRecIdx, UInt numCatEntries)
{
  ListPtr listPtr;
  ControlPtr popupPtr;
  FormPtr formPtr;

  /* fill the list */
  listPtr = DetailFormFillPopupList(listID, popupID, catRecIdx, numCatEntries, false);
  /* make the list and the matching popup display the first list item */
  formPtr = FrmGetFormPtr(formID_tsDetail);
  LstSetTopItem(listPtr, 0);
  popupPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, popupID));
  CtlSetLabel(popupPtr, LstGetSelectionText(listPtr, 0));
  return listPtr;
}

/******************************************************************
 * User has entered new entry details and clicked ok, time to add a
 * new entry to present day.
 ******************************************************************/
UInt
DetailFormNewEntry(void)
{
  Byte numEntries;
  UInt entryIdx;
  UInt attr;
  Err error;
  VoidHand dayHand;
  VoidHand entryHand;
  TSDayRecType day;
  TSDayRecType *dayPtr;
  TSEntryRecType *entryPtr;

  /* is there an existing record for this day? */
  if (!TSPresentDayRecExists)
    {
      /* no, so create one before creating the initial entry for the day */
      dayHand = DmNewRecord(TSDatabase, &TSPresentDayRecIdx, sizeof(TSDayRecType));
      /* fill the day record with appropriate details */
      dayPtr = MemHandleLock(dayHand);
      day.dayDate.day = TSPresentDayDate.day;
      day.dayDate.month = TSPresentDayDate.month;
      day.dayDate.year = TSPresentDayDate.year;
      day.numEntries = 1;       /* just about to add first record */
      numEntries = 1;
      DmWrite(dayPtr, 0, &day, sizeof(TSDayRecType));
      MemHandleUnlock(dayHand);
      DmReleaseRecord(TSDatabase, TSPresentDayRecIdx, true);
      /* mark the record as a day record */
      attr = TSDayRecAttr;
      error = DmSetRecordInfo(TSDatabase, TSPresentDayRecIdx, &attr, NULL);
      /* created new day record */
      TSPresentDayRecExists = true;
      /* set the next new entry index to the first record after this day */
      entryIdx = TSPresentDayRecIdx + 1;
      /* Check if we need to update the entry timer record index */
      if (TSEntryTimer.recIdx >= TSPresentDayRecIdx)
      {
        TSEntryTimer.recIdx += 1;
      }
    }
  else
    {
      /* yes, update the present day and find the point to insert the next entry record for the day */
      dayHand = DmGetRecord(TSDatabase, TSPresentDayRecIdx);
      dayPtr = MemHandleLock(dayHand);
      numEntries = dayPtr->numEntries;
      numEntries += 1;
      DmWrite(dayPtr, 0, &numEntries, sizeof(Byte));
      entryIdx = TSPresentDayRecIdx + numEntries;
      MemHandleUnlock(dayHand);
      DmReleaseRecord(TSDatabase, TSPresentDayRecIdx, true);
    }
  /* create a new database record */
  DmNewRecord(TSDatabase, &entryIdx, sizeof(TSEntryRecType) + 1);
  DmReleaseRecord(TSDatabase, entryIdx, true);
  /* mark the record as a day record */
  attr = TSEntryRecAttr;
  error = DmSetRecordInfo(TSDatabase, entryIdx, &attr, NULL);
  /* fill the entry record with details from the database */
  DetailFormToEntryRec(entryIdx);
  /* fill in the new record's day entry index */
  entryHand = DmGetRecord(TSDatabase, entryIdx);
  entryPtr = MemHandleLock(entryHand);
  DmWrite(entryPtr, (VoidPtr) & (entryPtr->entryNum) - (VoidPtr) entryPtr, &numEntries, sizeof(Byte));
  MemHandleUnlock(entryHand);
  DmReleaseRecord(TSDatabase, entryIdx, true);
  /* make sure the new entry will be visible in the main table */
  if (TSTableTopEntry + TSMainFormTableRows < numEntries)
    {
      /* need to scroll table up to show added entry */
      TSTableTopEntry = numEntries - TSMainFormTableRows;
    }
  /* Check if we need to update the entry timer record index */
  if (TSEntryTimer.recIdx >= entryIdx)
  {
    TSEntryTimer.recIdx += 1;
  }
  /* Return the new entry's database index */
  return entryIdx;
}

/******************************************************************
 * Handle scrolling the desc field on the detail form, driven by
 * scrollbar events.
 ******************************************************************/
void
DetailFormScroll(FormPtr formPtr, Int lines)
{
  Word blankLines;
  Short min;
  Short max;
  Short value;
  Short pageSize;
  FieldPtr fld;
  ScrollBarPtr bar;

  fld = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsDesc));

  if (lines < 0)
    {
      blankLines = FldGetNumberOfBlankLines(fld);
      FldScrollField(fld, -lines, winUp);

      /* If there were blank lines visible at the end of the field
       * then we need to update the scroll bar. */
      if (blankLines)
        {
          /* Update the scroll bar. */
          bar = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, scrollbarID_tsDesc));
          SclGetScrollBar(bar, &value, &min, &max, &pageSize);
          if (blankLines > -lines)
            max += lines;
          else
            max -= blankLines;
          SclSetScrollBar(bar, value, min, max, pageSize);
        }
    }
  else if (lines > 0)
    FldScrollField(fld, lines, winDown);
}

/******************************************************************
 * Update the form's scroll bar to reflect changes in the desc field.
 ******************************************************************/
void
DetailFormScrollBarUpdate(FormPtr formPtr)
{
  Word scrollPos;
  Word textHeight;
  Word fieldHeight;
  Short maxValue;
  FieldPtr fld;
  ScrollBarPtr bar;

  fld = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsDesc));
  bar = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, scrollbarID_tsDesc));

  FldGetScrollValues(fld, &scrollPos, &textHeight, &fieldHeight);

  if (textHeight > fieldHeight)
    maxValue = textHeight - fieldHeight;
  else if (scrollPos)
    maxValue = scrollPos;
  else
    maxValue = 0;

  SclSetScrollBar(bar, scrollPos, 0, maxValue, fieldHeight - 1);
}

/******************************************************************
 * Fills the supplied database entry record with fields from the
 * presently visible DetailForm.
 ******************************************************************/
void
DetailFormToEntryRec(UInt entryRecIdx)
{
  UInt recAttr;
  UInt descLen;
  ULong entrySize;
  ULong newEntrySize;
  TSEntryRecType entry;
  FormPtr formPtr;
  ControlPtr controlPtr;
  VoidHand hand;
  VoidPtr ptr;
  Byte hours;
  Byte mins;
  Byte clientSelectIdx;
  Byte projectSelectIdx;
  Byte taskSelectIdx;

  /* build a local entry from the GUI control settings */
  formPtr = FrmGetActiveForm();
  /* save the hour setting */
  DetailFormGetDuration(formPtr, &hours, &mins);
  entry.hours = hours + (mins << 4);
  /* save the selected client (if any) */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsClient));
  /* don't save the actual selection idx, save the translation table entry that contains the selection idx instead */
  hand = DmQueryRecord(TSDatabase, 1);
  ptr = MemHandleLock(hand);
  clientSelectIdx = (Byte) LstGetSelection((ListPtr) controlPtr);
  entry.clientIdx = CatReverseLookup(clientSelectIdx, ptr, TSMaxCatEntries);
  MemHandleUnlock(hand);
  /* save the project setting (if any) */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsProject));
  /* don't save the actual selection idx, save the translation table entry that contains the selection idx instead */
  hand = DmQueryRecord(TSDatabase, 2);
  ptr = MemHandleLock(hand);
  projectSelectIdx = (Byte) LstGetSelection((ListPtr) controlPtr);
  entry.projectIdx = CatReverseLookup(projectSelectIdx, ptr, TSMaxCatEntries);
  MemHandleUnlock(hand);
  /* save the task setting (if any) */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsTask));
  /* don't save the actual selection idx, save the translation table entry that contains the selection idx instead */
  hand = DmQueryRecord(TSDatabase, 3);
  ptr = MemHandleLock(hand);
  taskSelectIdx = (Byte) LstGetSelection((ListPtr) controlPtr);
  entry.taskIdx = CatReverseLookup(taskSelectIdx, ptr, TSMaxCatEntries);
  MemHandleUnlock(hand);
  /* if this is a new entry, save the client/project/task selection index for the auto categories option
   * NOTE: we *always* save this info for new entries, but we don't always use it when we init the new detail form */
  if (TSEditEntryIdx == 0)
    {
      /* new entry, save selected client/project/task selection index (NOT the translated index!) into app prefs */
      TSAppPrefs.newCatEntryIdx[0] = entry.clientIdx;
      TSAppPrefs.newCatEntryIdx[1] = entry.projectIdx;
      TSAppPrefs.newCatEntryIdx[2] = entry.taskIdx;
      /* also save the hour/minutes duration into app prefs */
      TSAppPrefs.newCatHours = entry.hours;
    }
  /* find out how big the entry will be (depends on the amount of descriptive text that has been entered by the user) */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsDesc));
  descLen = FldGetTextLength((FieldPtr) controlPtr);
  /* resize the record if required to accomodate the entered text */
  hand = DmGetRecord(TSDatabase, entryRecIdx); // marks as busy
  entrySize = MemHandleSize(hand);
  newEntrySize = sizeof(TSEntryRecType) + ((descLen + 1) * sizeof(Byte));
  if (entrySize != newEntrySize)
    {
      /* resize the record (NOTE: handle may change!) */
      DmReleaseRecord(TSDatabase, entryRecIdx, false); // clear busy bit
      hand = DmResizeRecord(TSDatabase, entryRecIdx, newEntrySize); // ? effect on busy bit??
      hand = DmGetRecord(TSDatabase, entryRecIdx); // mark as busy again??
      // the above clumsy hack [jo] 13-8-2005 seems to prevent problems below
    }
  /* write the entry */
  ptr = MemHandleLock(hand);
  /* save the current entry num */
  entry.entryNum = ((TSEntryRecType *) ptr)->entryNum;
  DmWrite(ptr, 0, &entry, sizeof(TSEntryRecType));
  /* write the descriptive text (if any) */
  if (descLen > 0)
    {
      /* write some actual text */
      DmWrite(ptr, sizeof(TSEntryRecType), FldGetTextPtr((FieldPtr) controlPtr), (descLen + 1) * sizeof(Byte));
    }
  else
    {
      /* make sure a null string gets in there */
      DmSet(ptr, sizeof(TSEntryRecType), 1, 0);
    }
  /* entry has been updated */
  MemHandleUnlock(hand);
  DmReleaseRecord(TSDatabase, entryRecIdx, true);
  // the above gives trouble [jo] 13-8-2005
  
  /* set the high bit of the record category (low four bits of attribute) on if entry is chargeable */
  DmRecordInfo(TSDatabase, entryRecIdx, &recAttr, NULL, NULL);
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsCharge));
  if (CtlGetValue(controlPtr))
    {
      /* turn bit 4 on (high bit of category) */
      recAttr = TSEntryRecAttr | 0x0008;
    }
  else
    {
      /* turn bit 4 off (high bit of category) */
      recAttr = TSEntryRecAttr & 0xFFF7;
    }
  DmSetRecordInfo(TSDatabase, entryRecIdx, &recAttr, NULL);
}

/******************************************************************
 * Detail form utility function to get duration hours/mins (in five mins intervals)
 ******************************************************************/
void DetailFormGetDuration(FormPtr formPtr, Byte* hoursPtr, Byte* fiveMinsPtr)
{
  *hoursPtr = LstGetSelection(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsHour)));
  *fiveMinsPtr = LstGetSelection(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsMin)));
}

/******************************************************************
 * Detail form utility function to set duration hours/mins (in five mins intervals)
 ******************************************************************/
void DetailFormSetDuration(FormPtr formPtr, Byte hours, Byte fiveMins)
{
  LstSetSelection(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsHour)), hours);
  CtlSetLabel(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listpopupID_tsHour)),
      LstGetSelectionText(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsHour)),
                          hours));
  LstSetSelection(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsMin)), fiveMins);
  CtlSetLabel(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listpopupID_tsMin)),
              LstGetSelectionText(FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsMin)),
                          fiveMins));
}

/*** End of section! ***/


// ============================================================================ //
// Database:
// #include "Database.c"

/******************************************************************
 * Application database related functions.
 ******************************************************************/

/******************************************************************
 *
 ******************************************************************/
Int
_SummaryFindCompareFunc(VoidPtr rec1Ptr, VoidPtr rec2Ptr, Int other_ignored)
{
  TSSummaryRecType *summaryRec1Ptr;
  TSSummaryRecType *summaryRec2Ptr;
  Int retVal;

  /* Start fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_PROLOGUE;

  /* Translate the generic rec pointers into a more useful form. */
  summaryRec1Ptr = rec1Ptr;
  summaryRec2Ptr = rec2Ptr;
  /* Order records by Client / Project / Task indexes */
  if (summaryRec1Ptr->catIdx[0] == summaryRec2Ptr->catIdx[0])
    {
      if (summaryRec1Ptr->catIdx[1] == summaryRec2Ptr->catIdx[1])
        {
          if (summaryRec1Ptr->catIdx[2] == summaryRec2Ptr->catIdx[2])
            {
              /* Records are equal */
              retVal = 0;
            }
          else
            {
              /* Order records by Task */
              retVal = summaryRec1Ptr->catIdx[2] - summaryRec2Ptr->catIdx[2];
            }
        }
      else
        {
          /* Order records by Project */
          retVal = summaryRec1Ptr->catIdx[1] - summaryRec2Ptr->catIdx[1];
        }
    }
  else
    {
      /* Order records by Client */
      retVal = summaryRec1Ptr->catIdx[0] - summaryRec2Ptr->catIdx[0];
    }
  /* End fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_EPILOGUE;

  return retVal;
}

/******************************************************************
 * Build a temporary summary database. Used to provide weekly and monthly summary views.
 ******************************************************************/
Boolean
BuildSummaryDatabase(DateType * startDate, DateType * endDate)
{
  Err error;
  LocalID summaryID;
  Int summaryCardNum;
  DmOpenRef summaryDB;
  UInt startDayRecIdx;
  UInt endDayRecIdx;
  UInt dayRecIdx;
  UInt summaryRecIdx;
  Boolean summaryRecExists;
  VoidHand entryHand;
  VoidHand summaryHand;
  VoidHand catHand;
  TSEntryRecType *entryPtr;
  TSSummaryRecType *summaryPtr;
  TSCatRecType *catPtr;
  TSSummaryRecType dummySummaryRec;
  TSSummaryTotalRecType summaryTotalRec;
  UInt recAttr;
  UInt numSummaryRecs;
  UInt sumHours;
  UInt sumMins;
  UInt catNum;

  /* Check if there's an existing summary database */
  summaryDB = DmOpenDatabaseByTypeCreator(TSSummaryDBType, TSAppID, dmModeReadWrite);
  if (summaryDB)
    {
      /* Get locaID, cardID etc. so we can delete database */
      DmOpenDatabaseInfo(summaryDB, &summaryID, NULL, NULL, &summaryCardNum, NULL);
      /* Close and delete existing summary database */
      DmCloseDatabase(summaryDB);
      DmDeleteDatabase(summaryCardNum, summaryID);
    }
  /* Create and open new summary database */
  error = DmCreateDatabase(0, TSSummaryDBName, TSAppID, TSSummaryDBType, false);
  if (error)
    {
      /* Failed to create database */
      if (error == dmErrMemError || error == memErrNotEnoughSpace)
        {
          FrmAlert(alertID_tsLowMem);
        }
      return false;
    }
  summaryDB = DmOpenDatabaseByTypeCreator(TSSummaryDBType, TSAppID, dmModeReadWrite);
  numSummaryRecs = 0;
  summaryTotalRec.totalHours = 0;
  summaryTotalRec.totalMins = 0;
  summaryTotalRec.totalChargeableHours = 0;
  summaryTotalRec.totalChargeableMins = 0;
  if (summaryDB)
    {
      /* Convert start/end dates into record indexes */
      if (!DateRangeToDatabaseIndex(startDate, endDate, &startDayRecIdx, &endDayRecIdx))
      {
          /* Nothing to summarise, so we've finished. */
          goto SUMMARY_FINISHED;
        }
      /* Build summary database */
      for (dayRecIdx = startDayRecIdx; dayRecIdx <= endDayRecIdx; dayRecIdx++)
        {
          /* Day or entry record? */
          DmRecordInfo(TSDatabase, dayRecIdx, &recAttr, NULL, NULL);
          if ((recAttr & TSRecAttrMask) == TSEntryRecAttr)
            {
              /* Have another entry record to include in summary database. */
              entryHand = DmQueryRecord(TSDatabase, dayRecIdx);
              entryPtr = MemHandleLock(entryHand);
              /* Build a dummy record to insert */
              MemMove(&dummySummaryRec, entryPtr, sizeof(Byte) * 3);    /* copy client/project/task */
              /* Translate the record's category indexes into actual indexes so summary is sorted alphabetically */
              for (catNum = 0; catNum < 3; catNum++)
                {
                  /* Get the appropriate category record */
                  catHand = DmQueryRecord(TSDatabase, catNum + 1);
                  catPtr = MemHandleLock(catHand);
                  /* Translate the index into an actual category entry */
                  dummySummaryRec.catIdx[catNum] = catPtr->transTable[dummySummaryRec.catIdx[catNum]];
                  MemHandleUnlock(catHand);
                }
              /* Translate entry Byte duration format to summary UInt duration format */
              dummySummaryRec.totalDuration = entryPtr->hours & 0x0F;   /* Hours in low 12 bits */
              dummySummaryRec.totalDuration |= (entryPtr->hours & 0xF0) << 8;   /* Minutes in high 4 bits */
              /* Add the existing record duration to the total duration */
              summaryTotalRec.totalHours += (dummySummaryRec.totalDuration & 0x0FFF);
              summaryTotalRec.totalMins += ((dummySummaryRec.totalDuration & 0xF000) >> 12);
              /* Roll up mins to hours */
              if (summaryTotalRec.totalMins >= 12)
                {
                  summaryTotalRec.totalMins -= 12;
                  summaryTotalRec.totalHours++;
                }
              if (recAttr & TSEntryChargedAttr)
                {
                  /* Add the existing record duration to the total chargeable duration */
                  summaryTotalRec.totalChargeableHours += (dummySummaryRec.totalDuration & 0x0FFF);
                  summaryTotalRec.totalChargeableMins += ((dummySummaryRec.totalDuration & 0xF000) >> 12);
                  /* Roll up mins to hours */
                  if (summaryTotalRec.totalChargeableMins >= 12)
                    {
                      summaryTotalRec.totalChargeableMins -= 12;
                      summaryTotalRec.totalChargeableHours++;
                    }
                }
              MemHandleUnlock(entryHand);
              /* Find the existing summary record, or the insert point for new summary record. */
              summaryRecIdx = DmFindSortPositionV10(summaryDB, &dummySummaryRec,
                               (DmComparF *) (&_SummaryFindCompareFunc), 0);
              /* Check if the summary record already exists or not */
              if (summaryRecIdx > 0 && (summaryRecIdx - 1) < numSummaryRecs)
                {
                  summaryHand = DmQueryRecord(summaryDB, summaryRecIdx - 1);
                  summaryPtr = MemHandleLock(summaryHand);
                  /* summaryRecExists set to true if the record already exists. */
                  if (summaryPtr->catIdx[0] == dummySummaryRec.catIdx[0]
                      && summaryPtr->catIdx[1] == dummySummaryRec.catIdx[1]
                      && summaryPtr->catIdx[2] == dummySummaryRec.catIdx[2])
                    {
                      /* Existing summary record just BEFORE returned summary record index */
                      summaryRecIdx--;
                      summaryRecExists = true;
                    }
                  else
                    {
                      summaryRecExists = false;
                    }
                  MemHandleUnlock(summaryHand);
                }
              else
                {
                  /* Need to append new summary record to end of summary database. */
                  summaryRecExists = false;
                }
              /* Update existing record or create new record? */
              if (summaryRecExists)
                {
                  /* Update existing record. */
                  summaryHand = DmGetRecord(summaryDB, summaryRecIdx); // [a]
                  summaryPtr = MemHandleLock(summaryHand);
                  /* Add the existing record duration to the dummy record duration */
                  sumHours = (summaryPtr->totalDuration & 0x0FFF)
                    + (dummySummaryRec.totalDuration & 0x0FFF);
                  sumMins = ((summaryPtr->totalDuration & 0xF000) >> 12)
                    + ((dummySummaryRec.totalDuration & 0xF000) >> 12);
                  /* Roll up mins to hours */
                  if (sumMins >= 12)
                    {
                      sumMins -= 12;
                      sumHours++;
                    }
                  /* Store total duration in dummy record */
                  dummySummaryRec.totalDuration = (sumMins << 12);
                  dummySummaryRec.totalDuration |= sumHours;
                  /* Copy in the updated dummy record */
                  DmWrite(summaryPtr, 0, &dummySummaryRec, sizeof(TSSummaryRecType));
                }
              else
                {
                  /* Create new record. */
                  summaryHand = DmNewRecord(summaryDB, &summaryRecIdx, sizeof(TSSummaryRecType));
                  if (!summaryHand)
                    {
                      /* Get locaID, cardID etc. so we can delete database as we appear to be low on memory */
                      DmOpenDatabaseInfo(summaryDB, &summaryID, NULL, NULL, &summaryCardNum, NULL);
                      /* Close and delete existing summary database */
                      DmCloseDatabase(summaryDB);
                      DmDeleteDatabase(summaryCardNum, summaryID);
                      /* Alert the user the operation failed */
                      FrmAlert(alertID_tsLowMem);
                      return false;
                    }
                  summaryPtr = MemHandleLock(summaryHand);
                  /* Copy in the dummy record */
                  DmWrite(summaryPtr, 0, &dummySummaryRec, sizeof(TSSummaryRecType));
                  numSummaryRecs++;
                }
              MemHandleUnlock(summaryHand);
              DmReleaseRecord(summaryDB, summaryRecIdx, true);
            }
        }
      /* Completed summary database */
    SUMMARY_FINISHED:
      /* Add total hours/minutes record to end of database */
      summaryHand = DmNewRecord(summaryDB, &numSummaryRecs, sizeof(TSSummaryTotalRecType)); // [b]
      if (!summaryHand)
        {
          /* Get locaID, cardID etc. so we can delete database as we appear to be low on memory */
          DmOpenDatabaseInfo(summaryDB, &summaryID, NULL, NULL, &summaryCardNum, NULL);
          /* Close and delete existing summary database */
          DmCloseDatabase(summaryDB);
          DmDeleteDatabase(summaryCardNum, summaryID);
          /* Alert the user the operation failed */
          FrmAlert(alertID_tsLowMem);
          return false;
        }
      /* Save the summary database total hours and minutes */
      summaryPtr = MemHandleLock(summaryHand);  /* Just reusing pointer */
      DmWrite(summaryPtr, 0, &summaryTotalRec, sizeof(TSSummaryTotalRecType));
      MemHandleUnlock(summaryHand);
      
      DmReleaseRecord(summaryDB, numSummaryRecs, true); 
      // [jo] fix the error caused by [b] 13-8-2005 (See below). 
      
      DmCloseDatabase(summaryDB);
      /// [jo] Error was here. Records left busy in 'closed unprotected' database!
      /// Look at all DmGetRecord and DmNewRecord invocations and see if
      /// corresponding DmReleaseRecord stmts exist ==>
      /// [a] summaryRecIdx seems ok.
      /// [b] problem was here.
      return true;
    }
  else
    {
      /* Failed to open created summary database for some reason (we'll assume it's low memory) */
      FrmAlert(alertID_tsLowMem);
      return false;
    }
}

/******************************************************************
 * Return the number of summary database records (if any).
 * Note the returned count includes the last 'totals' record in an
 * existing summary database.
 ******************************************************************/
Int GetNumSummaryRecords(void)
{
  DmOpenRef summaryDB;
  Int numSummaryRecs;

  /* Check if there's an existing summary database */
  summaryDB = DmOpenDatabaseByTypeCreator(TSSummaryDBType, TSAppID, dmModeReadWrite);
  if (summaryDB)
    {
      /* Get the number of records in the database */
      numSummaryRecs = DmNumRecords(summaryDB);
      DmCloseDatabase(summaryDB);
    }
  else
    {
      /* No records to display */
      numSummaryRecs = 0;
    }
  return numSummaryRecs;
}

/******************************************************************
 * Convert a start and end date into indexes in the Timesheet database.
 * Returns false if no actual records fall within range.
 * Returns true if at least one day record falls within range.
 ******************************************************************/
Boolean DateRangeToDatabaseIndex(DateType * startDate, DateType * endDate,
                                        UInt * startRecIdx, UInt * endRecIdx)
{
  Boolean startRecExists;
  Boolean endRecExists;
  VoidHand dayHand;
  TSDayRecType *dayPtr;

  /* Find start day index */
  FindDayRec(startDate, startRecIdx, &startRecExists);
  /* Find end day index */
  FindDayRec(endDate, endRecIdx, &endRecExists);
  /* Check if we have anything to do */
  if ((*startRecIdx) == (*endRecIdx) && !(startRecExists || endRecExists))
    {
      /* No records fall between start and end date */
      return false;
    }
  /* Find the adjusted endRecIdx */
  if (endRecExists)
    {
      /* Find number of entries in existing end day */
      dayHand = DmQueryRecord(TSDatabase, (*endRecIdx));
      dayPtr = MemHandleLock(dayHand);
      (*endRecIdx) += dayPtr->numEntries;
      MemHandleUnlock(dayHand);
    }
  else
    {
      /* End day doesn't exist, go to end of previous day. */
      (*endRecIdx)--;
    }
  /* Have at least one day of records in range */
  return true;
}

/******************************************************************
 * Erase all database records that fall between start and end date inclusive
 ******************************************************************/
void
EraseDatabaseRange(DateType * startDate, DateType * endDate)
{
  UInt startRecIdx;
  UInt endRecIdx;
  UInt recIdx;

  /* Convert start and end dates into database record indexes */
  if (DateRangeToDatabaseIndex(startDate, endDate, &startRecIdx, &endRecIdx))
    {
      /* Have some records to delete */
      for(recIdx = startRecIdx; recIdx <= endRecIdx; recIdx ++)
        {
          /* That's not a typo, we just keep deleting the start record and following records
           * get shuffled up. */
          DmRemoveRecord(TSDatabase, startRecIdx);
        }
      /* If we're timing and entry, check if we just deleted it */
      if (TSEntryTimer.recIdx > 0)
      {
        if (TSEntryTimer.recIdx >= startRecIdx)
        {
          if (TSEntryTimer.recIdx <= endRecIdx)
          {
            /* Timed entry has been deleted, clear timer */
            TSEntryTimer.recIdx = 0;
            TSEntryTimer.secs = 0;
          }
          else
          {
            /* Timed entry has not been deleted, but does need to be adjusted down as
             * intermediate records have been deleted */
            TSEntryTimer.recIdx -= (endRecIdx - startRecIdx) + 1;
          }
        }
        /* else do nothing, timed record appears BEFORE start deletion index */
      }
    }
}

/******************************************************************
 * Record compare function used by the DmFindSortPosition function.
 ******************************************************************/
Int
_DayFindCompareFunc(VoidPtr rec1Ptr, VoidPtr rec2Ptr, Int other_ignored,
                    SortRecordInfoPtr rec1InfoPtr, SortRecordInfoPtr rec2InfoPtr, VoidHand appInfoH)
{
  Int retVal;
  Byte rec1Type;
  Byte rec2Type;
  VoidHand rec2Hand;
  Long UID;
  UInt rec2ID;
  
  /* Start fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_PROLOGUE;

  rec2ID = 0;
  rec2Hand = 0; ///
  /* find the Timesheet record types of the two records we've been passed. */
  rec1Type = rec1InfoPtr->attributes & TSRecAttrMask;
  rec2Type = rec2InfoPtr->attributes & TSRecAttrMask;
  /* act depending on the record's first two attribute bits. These bits uniquely identify all record types
   * presently used in the Timesheet database */
  if (rec2Type == TSSpecialRecAttr)
    {
      /* special record in the database (always come 'first') */
      retVal = 1;
    }
  else
    {
      if (rec2Type == TSEntryRecAttr)
        {
          /* day entry records in the database, must find the earlier DAY header record this entry corresponds to */
          /* find the ID this record corresponds to */
          UID = (rec2InfoPtr->uniqueID[2])
            | ((Long) rec2InfoPtr->uniqueID[1] << 8)
            | ((Long) rec2InfoPtr->uniqueID[0] << 16);
          DmFindRecordByID(TSDatabase, UID, &rec2ID);
          /* search back to get the day this entry lives under */
          rec2ID -= ((TSEntryRecType *) rec2Ptr)->entryNum;
          /* load and lock the day record */
          rec2Hand = DmQueryRecord(TSDatabase, rec2ID);
          rec2Ptr = MemHandleLock(rec2Hand);
        }
      /* have two day headers to compare, so compare them and set the return result */
      retVal = DateToDays(((TSDayRecType *) rec1Ptr)->dayDate) - DateToDays(((TSDayRecType *) rec2Ptr)->dayDate);
      /* release any records we had to search for */
      if (rec2ID != 0)
        {
          MemHandleUnlock(rec2Hand);
        }
    }

  /* end fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_EPILOGUE;

  return retVal;
}

/******************************************************************
 *
 ******************************************************************/
Boolean FindEarliestDayRec(void)
{
  VoidHand recHand;
  TSDayRecType *recPtr;

  /* Check the database has at least one day in it */
  if (DmNumRecords(TSDatabase) > TSFirstDayRecIdx)
    {
      /* Change the current date to the earliest (first) day in the database */
      recHand = DmQueryRecord(TSDatabase, TSFirstDayRecIdx);
      recPtr = MemHandleLock(recHand);
      TSPresentDayDate.day = recPtr->dayDate.day;
      TSPresentDayDate.month = recPtr->dayDate.month;
      TSPresentDayDate.year = recPtr->dayDate.year;
      MemHandleUnlock(recHand);
      /* Found earliest day */
      return true;
    }
  else
    {
      /* No earliest day */
      return false;
    }
}

/******************************************************************
 *
 ******************************************************************/
Boolean
FindLatestDayRec(void)
{
  VoidHand recHand;
  TSDayRecType *recPtr;

  /* Check the database has at least one day in it */
  if (DmNumRecords(TSDatabase) > TSFirstDayRecIdx)
    {
      /* Change the current date to the latest (last) day in the database */
      recHand = DmQueryRecord(TSDatabase, FindPrevDayRecIdx(TSDatabase, DmNumRecords(TSDatabase) - 1));
      recPtr = MemHandleLock(recHand);
      TSPresentDayDate.day = recPtr->dayDate.day;
      TSPresentDayDate.month = recPtr->dayDate.month;
      TSPresentDayDate.year = recPtr->dayDate.year;
      MemHandleUnlock(recHand);
      return true;
    }
  else
    {
      /* No later day */
      return false;
    }
}

/******************************************************************
 * Finds TSPresentDayRecIdx for the day corresponds to the date
 * held in TSPresentDayDate using the Data Manager binary search.
 * Also sets up TSPresentDayRecIdx to the appropriate value.
 ******************************************************************/
void
FindDayRec(DateType * dayDate, UInt * presentDayRecIdx, Boolean * presentDayRecExists)
{
  TSDayRecType dummyDayRec;
  SortRecordInfoType dummyDayInfo;
  VoidHand dayRecHand;
  TSDayRecType *dayRecPtr;
  UInt searchRecIdx;

  /* Check if we have any records to search */
  if (DmNumRecords(TSDatabase) == TSFirstDayRecIdx)
    {
      /* Nope, don't bother searching */
      *presentDayRecIdx = TSFirstDayRecIdx;
      *presentDayRecExists = false;
      return;
    }
  /* set up the 'new' record for searching */
  MemMove(&(dummyDayRec.dayDate), dayDate, sizeof(DateType));
  dummyDayInfo.attributes = TSDayRecAttr;
  /* Use Data Manager's binary search. Requires a moderately intelligent comparison function due to database structure. */
  searchRecIdx = DmFindSortPosition(TSDatabase, &dummyDayRec, &dummyDayInfo, (DmComparF *) (&_DayFindCompareFunc), 0);
  /* Record may or may not already exist, have to perform one more test to see what search returned. */
  /* Find the day entry the found record belongs in */
  *presentDayRecIdx = FindPrevDayRecIdx(TSDatabase, searchRecIdx - 1);
  /* Load and lock... */
  dayRecHand = DmQueryRecord(TSDatabase, *presentDayRecIdx);
  dayRecPtr = MemHandleLock(dayRecHand);
  /* does it match the search date? */
  if (DateToDays(dayRecPtr->dayDate) != DateToDays(*dayDate))
    {
      /* revert to original returned index */
      *presentDayRecIdx = searchRecIdx;
      *presentDayRecExists = false;
    }
  else
    {
      /* found matching day */
      *presentDayRecExists = true;
    }
  MemHandleUnlock(dayRecHand);
}

/******************************************************************
 * Find the previous day record index from a given record index.
 * Note database to search is specified as argument because this
 * function is used by the FindInDatabase function (which is invoked
 * by the Palmpilot 'Find' button).
 ******************************************************************/
UInt
FindPrevDayRecIdx(DmOpenRef db, UInt presentRecIdx)
{
  VoidHand entryHand;
  TSEntryRecType *entryPtr;
  UInt attr;

  /* sanity check required for binary search */
  if (presentRecIdx < TSFirstDayRecIdx)
    {
      return TSFirstDayRecIdx;
    }
  /* first check if we're starting on a day */
  DmRecordInfo(db, presentRecIdx, &attr, NULL, NULL);
  if ((attr & 0x07) == TSDayRecAttr)
    {
      /* already on a day! */
      return presentRecIdx;
    }
  /* go back to the day record for the current entry */
  entryHand = DmQueryRecord(db, presentRecIdx);
  entryPtr = MemHandleLock(entryHand);
  presentRecIdx -= entryPtr->entryNum;
  MemHandleUnlock(entryHand);
  return presentRecIdx;
}

/******************************************************************
 * Retrieve the current application preferences from the opened Timesheet database.
 ******************************************************************/
Err OpenPreferences(void)
{
  VoidHand hand;
  TSAppPrefType *prefPtr;

  /* Retrieve the application preference info from record zero of the database */
  hand = DmQueryRecord(TSDatabase, 0);

//  SHOWERROR("The handle", (Int32)hand);

  prefPtr = MemHandleLock(hand);
  /* Load global copy of application preferences */
  MemMove(&TSAppPrefs, prefPtr, sizeof(TSAppPrefType));
  /* Check if we have a timer record on the end */
  if (MemHandleSize(hand) >= sizeof(TSAppPrefType) + sizeof(TSEntryTimerType))
  {
    /* Load global timer record as well */
    MemMove(&TSEntryTimer, ((void*)prefPtr) + sizeof(TSAppPrefType), sizeof(TSEntryTimerType));
    MemHandleUnlock(hand);
  }
  else
  {
    /* No timer record, so clear global timer record */
    MemSet(&TSEntryTimer, sizeof(TSEntryTimerType), 0);
    /* Resize the preferences record to include at least one timer structure */
    MemHandleUnlock(hand);
    hand = DmResizeRecord(TSDatabase, 0, sizeof(TSAppPrefType) + sizeof(TSEntryTimerType));
    if (!hand)
    {
      /* Failed to resize record */
      FrmAlert(alertID_tsLowMem);
      return appErrorClass;
    }
///    DmReleaseRecord(TSDatabase, 0, false);
  /// NOO. It's wrong to release record if only DmQuery! [jo] 12/8/2005
  }
  /* Opened preferences ok */
  return 0;
}

/******************************************************************
 * Save the current application preferences back to the open Timesheet database, typically
 * prior to application close.
 ******************************************************************/
void ClosePreferences(void)
{
  VoidHand hand;
  VoidPtr voidPtr;

  /* Save the application preferences back to the timesheet resource database */
  hand = DmGetRecord(TSDatabase, 0);
  voidPtr = MemHandleLock(hand);
  DmWrite(voidPtr, 0, &TSAppPrefs, sizeof(TSAppPrefType));
  /* Sanity check the timer */
  if (TSEntryTimer.recIdx == 0 && TSEntryTimer.secs > 0)
  {
        /* HACK! Timing non-saved entry, stop timer */
        TSEntryTimer.recIdx = 0;
        TSEntryTimer.secs = 0;
  }
  /* If we get this far we can be sure there's room for at least one timer record in the
   * preferences record as OpenPrefences will have expanded the prefs record if required */
  DmWrite(voidPtr, sizeof(TSAppPrefType), &TSEntryTimer, sizeof(TSEntryTimerType));
  MemHandleUnlock(hand);
  DmReleaseRecord(TSDatabase, 0, true);
}

/******************************************************************
 * Used by the FindInDatabase function below to search entry categories for matching
 * client/project/task names.
 ******************************************************************/
Boolean
FindInCategory(DmOpenRef db, UInt catRec, UInt entryCatIdx, FindParamsPtr findParams)
{
  Boolean match;
  VoidHand catHand;
  TSCatRecType *catPtr;
  CharPtr catDescPtr;
  Word pos;

  catHand = DmQueryRecord(db, catRec);
  catPtr = MemHandleLock(catHand);
  catDescPtr = catPtr->cats[catPtr->transTable[entryCatIdx]];
  match = FindStrInStr(catDescPtr, findParams->strToFind, &pos);
  MemHandleUnlock(catHand);
  return match;
}

/******************************************************************
 * Search the Timesheet database for matching records. Called by PilotMain
 * to implement the Find command. Basically lifted from Memopad.c, thanks Palm computing!
 ******************************************************************/
void
FindInDatabase(FindParamsPtr findParams)
{
  Boolean done;
  Boolean fits;
  Int textLen;
  Int width;
  UInt recIdx;
  UInt numRecs;
  UInt recAttr;
  UInt cardNo;
  LocalID dbID;
  Word pos;
  VoidHand recordHand;
  VoidPtr recordPtr;
  Boolean match;
  CharPtr entryDescPtr;
  DmOpenRef tsDatabase;
  DateType dayDate;
  FontID currFont;
  RectangleType rect;
  /// Char dateStr[20 + 1];
  /// [jo]: hmm I think we need to fix this one, thus:
  Char dateStr[dowLongDateStrLength]; ///
  SystemPreferencesType sysPrefs;

  /* Get a copy of the system's preferences */
  PrefGetPreferences(&sysPrefs);
  /* Mark dayDate as unfilled (yet). Required as we may begin searching from middle of entries for certain day, so
   * won't have day date for any matching entries, see code below. */
  dayDate.day = 0;
  /* Find the app's database. */
  tsDatabase = DmOpenDatabaseByTypeCreator(TSDBType, TSAppID, findParams->dbAccesMode);
  if (!tsDatabase)
    {
      /* Return without anything, also indicate that no more records are expected. */
      findParams->more = false;
      return;
    }
  numRecs = DmNumRecords(tsDatabase);
  DmOpenDatabaseInfo(tsDatabase, &dbID, 0, 0, &cardNo, 0);
  /* Display the heading line */
  recordHand = DmGetResource(strRsc, stringID_tsFindHeader);
  recordPtr = MemHandleLock(recordHand);
  done = FindDrawHeader(findParams, (CharPtr) recordPtr);
  MemHandleUnlock(recordHand);
  if (done)
    {
      /* There was no more room to display the heading line */
      DmCloseDatabase(tsDatabase);
      return;
    }
  /* Search the Timesheet entries for the string, starting with the recIdx passed.
   * This allows the search code to be called multiple times when more than one screen
   * of records are found. */
  recIdx = findParams->recordNum;
  /* Skip the first 4 database records as they contain app preferences and category information. */
  if (recIdx < TSFirstDayRecIdx)
    {
      recIdx = TSFirstDayRecIdx;
      findParams->recordNum = TSFirstDayRecIdx;
    }
  while (true)
    {
      /* Because applications can take a long time to finish a find when
       * the result may be on the screen or for other reasons, users like
       * to be able to stop the find.  Stop the find if an event is pending.
       * This stops if the user does something with the device.  Because
       * this call slows down the search we perform it every so many
       * records instead of every record.  The response time should still
       * be short without introducing much extra work to the search.
       * Every 16 records is guessed at to check often enough to respond
       * quickly to the user without wasting time checking too often. */
      if ((recIdx & 0x000f) == 0 && EvtSysEventAvail(true))
        {
          /* Stop the search process. */
          findParams->more = true;
          break;
        }
      /* Have we run out of records? */
      if (recIdx >= numRecs)
        {
          findParams->more = false;
          break;
        }
      /* Get the next record attributes. */
      DmRecordInfo(tsDatabase, recIdx, &recAttr, NULL, NULL);
      recordHand = DmQueryRecord(tsDatabase, recIdx);
      recordPtr = MemHandleLock(recordHand);
      if ((recAttr & 0x07) == TSDayRecAttr)
        {
          /* On day record, don't do any searching, but save the date of the latest day reached. */
          dayDate.day = ((TSDayRecType *) recordPtr)->dayDate.day;
          dayDate.month = ((TSDayRecType *) recordPtr)->dayDate.month;
          dayDate.year = ((TSDayRecType *) recordPtr)->dayDate.year;
          MemHandleUnlock(recordHand);
        }
      else
        {
          /* On entry record, search the categories and entry description for matching string. */
          /* Check the client entry */
          match = FindInCategory(tsDatabase, 1, ((TSEntryRecType *) recordPtr)->clientIdx, findParams);
          /* Check the project entry */
          match |= FindInCategory(tsDatabase, 2, ((TSEntryRecType *) recordPtr)->projectIdx, findParams);
          /* Check the task entry */
          match |= FindInCategory(tsDatabase, 3, ((TSEntryRecType *) recordPtr)->taskIdx, findParams);
          /* Search for the string passed, if it's found display the day date of the entry.
           * An application should call FindStrInStr for each part of the record which
           * needs to be searched. */
          entryDescPtr = (CharPtr) (recordPtr + sizeof(TSEntryRecType));
          match |= FindStrInStr(entryDescPtr, findParams->strToFind, &pos);
          if (match)
            {
              /* Add the match to the find parameter block,  if there is no room to
               * display the match the following function will return true. */
              done = FindSaveMatch(findParams, recIdx, pos, 0, 0, cardNo, dbID);
              if (!done)
                {
                  /* Do we have a day date to display? */
                  if (dayDate.day == 0)
                    {
                      /* No, so find the date of the day record this entry corresponds to. */
                      /* Unload entry record first as FindPrevDayRecIdx queries it. */
                      MemHandleUnlock(recordHand);
                      recordHand = DmQueryRecord(tsDatabase, FindPrevDayRecIdx(tsDatabase, recIdx));
                      recordPtr = MemHandleLock(recordHand);
                      /* Save date from day record */
                      dayDate.day = ((TSDayRecType *) recordPtr)->dayDate.day;
                      dayDate.month = ((TSDayRecType *) recordPtr)->dayDate.month;
                      dayDate.year = ((TSDayRecType *) recordPtr)->dayDate.year;
                      MemHandleUnlock(recordHand);
                      /* Relock entry record and continue */
                      recordHand = DmQueryRecord(tsDatabase, recIdx);
                      recordPtr = MemHandleLock(recordHand);
                      entryDescPtr = (CharPtr) (recordPtr + sizeof(TSEntryRecType));
                    }
                  /* Get the bounds of the region where we will draw the results. */
                  FindGetLineBounds(findParams, &rect);
                  /* Display the record in the search dialog.  We move in one pixel
                   * so that when the record is inverted the left edge is solid. */
                  rect.topLeft.x++;
                  rect.extent.x--;
                  /* Set the standard font, save the current font. */
                  currFont = FntSetFont(stdFont);

                  /* Draw the date string first */
                  DateToAscii(dayDate.month, dayDate.day, dayDate.year + NINETEENTWENTY, sysPrefs.dateFormat, dateStr);
                  textLen = StrLen(dateStr);
                  width = FntCharsWidth(dateStr, textLen) + 4;
                  WinDrawChars(dateStr, textLen, rect.topLeft.x, rect.topLeft.y);
                  rect.topLeft.x += width;
                  rect.extent.x -= width;

                  /* Draw start of the entry description */
                  width = rect.extent.x - 2;
                  textLen = StrLen(entryDescPtr);
                  FntCharsInWidth(entryDescPtr, &width, &textLen, &fits);
                  /* Now draw the text from the record. */
                  WinDrawChars(entryDescPtr, textLen, rect.topLeft.x, rect.topLeft.y);
                  /* Restore the font. */
                  FntSetFont(currFont);
                  /* The line number needs to be increment since a line is used for the record. */
                  findParams->lineNumber++;
                }
            }
          MemHandleUnlock(recordHand);
          if (done)
            {
              break;
            }
        }
      recIdx += 1;
    }
  /* Finished searching the database */
  DmCloseDatabase(tsDatabase);
}

/******************************************************************
 * Initialise a new Timesheet database with a small set of
 * application resource records.
 ******************************************************************/
Err
InitDatabase(void)
{
  Err error;
  VoidHand hnd;
  Ptr ptr;
  UInt resId = 0;
  TSAppPrefType defaultPref =
  {
    {TSNumDefaultClients, TSNumDefaultProjects, TSNumDefaultTasks},
    {0, 0, 0}, TSPrefDefaultFlags, 0
    };

  /* create a new record 0 in the database (used to hold the application info) */
  hnd = DmNewRecord(TSDatabase, &resId, sizeof(TSAppPrefType));
  if (!hnd)
    {
      /* failed to allocate record */
      return appErrorClass;
    }
  ptr = MemHandleLock(hnd);
  DmWrite(ptr, 0, &defaultPref, sizeof(TSAppPrefType));
  MemHandleUnlock(hnd);
  DmReleaseRecord(TSDatabase, resId, true);
  /* create a new record in the database to hold the client list */
  resId += 1;
  error = InitCatResource(&resId, stringID_tsDefaultClients, TSNumDefaultClients);
  if (error)
    {
      /* failed to init record */
      return error;
    }
  /* create a new record in the database to hold the project list */
  error = InitCatResource(&resId, stringID_tsDefaultProjects, TSNumDefaultProjects);
  if (error)
    {
      /* failed to init record */
      return error;
    }
  /* create a new record in the database to hold the task list */
  error = InitCatResource(&resId, stringID_tsDefaultTasks, TSNumDefaultTasks);
  if (error)
    {
      /* failed to init record */
      return error;
    }
  /* resource database initialised */
  return 0;
}

/******************************************************************
 * Open and existing application database, or create a new one (with sensible defaults) if none is found.
 ******************************************************************/
Err
OpenDatabase(void)
{
  Err error = 0;
  LocalID dbID;
  UInt dbAttr;
  UInt dbVer;
  UInt cardNo;

  /* open or create the timesheet database for the day and entry records */
  TSDatabase = DmOpenDatabaseByTypeCreator(TSDBType, TSAppID, dmModeReadWrite);
  if (!TSDatabase)
    {
      /* failed to open existing database, so create an new database */
      error = DmCreateDatabase(0, TSDBName, TSAppID, TSDBType, false);
      if (error)
        {
          /* failed to create database */
          return error;
        }
      /* open the new database */
      TSDatabase = DmOpenDatabaseByTypeCreator(TSDBType, TSAppID, dmModeReadWrite);
      if (!TSDatabase)
        {
          /* failed to open database */
          return appErrorClass;
        }
      /* set the database attributes (version, creation date and backup flag) */
      DmOpenDatabaseInfo(TSDatabase, &dbID, NULL, NULL, &cardNo, NULL);
      dbAttr = dmHdrAttrBackup;
      dbVer = TSDBVersion;
      DmSetDatabaseInfo(cardNo, dbID, NULL, &dbAttr, &dbVer, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
      /* initialise the new database with resource records */
      error = InitDatabase();
      if (error)
        {
          /* failed to initialise opened database */
          DmCloseDatabase(TSDatabase);
          DmDeleteDatabase(cardNo, dbID);
          return error;
        }
    }
  else
    {
      /* version check the existing Timesheet database */
      if (VersionCheckDatabase() != 0)
        {
          /* existing database is incompatible */
          DmCloseDatabase(TSDatabase);
          return appErrorClass;
        }
    }
  /* Get the application global preferences from opened database */
  error = OpenPreferences();
  return error;
}

/******************************************************************
 * Compress the category translation table and re-map all existing
 * database entries if required. Return true if some reorging has
 * been performed (generating free category entries), or false if
 * the category is completely full (without any redundant entries).
 ******************************************************************/
Boolean
ReorgDatabase(TSCatRecType * catPtr, Int catRecIdx)
{
  Byte idx;
  Byte catIdx;
  Byte compressCount;
  UInt recIdx;
  UInt numEntries;
  Byte revTransTable[TSMaxCatEntries];
  Byte reorgTransTable[TSMaxCatEntries];
  Byte newTransTable[TSMaxCatEntries];
  VoidHand hand;
  TSDayRecType *dayPtr;
  TSEntryRecType *entryPtr;

  /* init reverse trans table, reorg trans table and new trans table */
  for (idx = 0; idx < TSMaxCatEntries; idx += 1)
    {
      revTransTable[idx] = TSMaxCatEntries;
      newTransTable[idx] = TSMaxCatEntries;
      reorgTransTable[idx] = 0;
    }
  /* build a reverse trans table */
  for (idx = 0, compressCount = 0; idx < TSMaxCatEntries; idx += 1)
    {
      catIdx = catPtr->transTable[idx];
      if (catIdx != TSMaxCatEntries)
        {
          if (revTransTable[catIdx] == TSMaxCatEntries)
            {
              /* nothing entered for this category yet, so fill it */
              revTransTable[catIdx] = idx;
            }
          else
            {
              /* this entry has already been filled, ignore (ie. squeeze out repeat entries) */
              compressCount += 1;
            }
        }                       /* else empty entry? why are we performing reorg? ignore */
    }
  /* have we compressed trans table at all? */
  if (compressCount == 0)
    {
      /* no, don't bother doing any more work, category is completely full. */
      return false;
    }
  /* build a reorg trans table */
  for (idx = 0; idx < TSMaxCatEntries; idx += 1)
    {
      /* mapping from old transTable index to new (compressed) transTable index */
      reorgTransTable[idx] = revTransTable[catPtr->transTable[idx]];
    }
  /* reorg existing transtable */
  for (catIdx = 0; catIdx < TSMaxCatEntries; catIdx += 1)
    {
      if (revTransTable[catIdx] != TSMaxCatEntries)
        {
          newTransTable[revTransTable[catIdx]] = catIdx;
        }
    }
  /* apply reorg trans table to database entries */
  recIdx = TSFirstDayRecIdx;
  while (recIdx < DmNumRecords(TSDatabase))
    {
      /* reorg every entry for current day record */
      hand = DmQueryRecord(TSDatabase, recIdx);
      dayPtr = MemHandleLock(hand);
      numEntries = dayPtr->numEntries;
      MemHandleUnlock(hand);
      for (recIdx += 1; numEntries > 0; numEntries -= 1, recIdx += 1)
        {
          hand = DmGetRecord(TSDatabase, recIdx);
          entryPtr = MemHandleLock(hand);
          catIdx = *(((BytePtr) entryPtr) + (catRecIdx - 1));
          DmWrite(entryPtr, (catRecIdx - 1) * sizeof(Byte), &reorgTransTable[catIdx], sizeof(Byte));
          MemHandleUnlock(hand);
          DmReleaseRecord(TSDatabase, recIdx, true);
        }
    }
  /* reorg the preference saved auto categories */
  TSAppPrefs.newCatEntryIdx[catRecIdx - 1] = reorgTransTable[TSAppPrefs.newCatEntryIdx[catRecIdx - 1]];
  /* save the updated translation table */
  DmWrite(catPtr, 0, newTransTable, sizeof(Byte) * TSMaxCatEntries);
  return true;
}

/******************************************************************
 * [jo] to save space we could clip out the lot. Simply fail if bad version!
 * Who is using earlier versions?? [done]
 * 
 ******************************************************************/
Err UpgradeDatabase(UInt cardNo, LocalID dbID, UInt fromDBVersion)
{
  // [jo] we brutally clipped out all of the fixes. Saves 1K of code.

 /* Don't know how to upgrade this database version! */
 return appErrorClass;

}

/******************************************************************
 * Check the version of an existing Timesheet database is compatible
 * with the database version the application expects. If not, attempt
 * to upgrade the database, or return an error if we can't.
 ******************************************************************/
Err
VersionCheckDatabase(void)
{
  Err error;
  LocalID dbID;
  UInt dbVer;
  UInt cardNo;

  /* get the database version number */
  DmOpenDatabaseInfo(TSDatabase, &dbID, NULL, NULL, &cardNo, NULL);
  DmDatabaseInfo(cardNo, dbID, NULL, NULL, &dbVer, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL);
  /* check the database version is compatible with the database version Timesheet expects */
  if (dbVer != TSDBVersion)
    {
      if (FrmAlert(alertID_tsDBConversion) == 1)
      {
        /* User cancelled dialog, exit Timesheet without upgrading database */
        return appErrorClass;
      }
      /* existing database differs from the version application requires */
      error = UpgradeDatabase(cardNo, dbID, dbVer);
      if (error)
        {
          FrmAlert(alertID_tsDBVerIncompat);
          return error;
        }
    }
  /* database looks good */
  return 0;
}

/*** End of section! ***/


// ============================================================================ //
// UIUtil:
// #include "UIUtil.c"

/******************************************************************
 * The day date has changed, update the active form appropriately (either detail or main)
 ******************************************************************/


// -------------------------------------------------------------- //
// FormUpdateDay: [jo: 12-8-2005: rewrite]

void FormUpdateDay(DateType * dayDate)
{
  ControlPtr controlPtr;
  FontID prevFont;
  FormPtr formPtr;
  RectangleType rect;
  
  Char * mydate;  // jo.
  Int16 length;
  
  Coord jox;
  Coord joy;

  mydate = (Char *) e_New(1+dowLongDateStrLength);  // NOT 21. We +1 but ? need this.
  // HARD CODING OF 21 CAUSED MEMORY ERROR!
  // could probably revert to his code with the small amendment. 12-8-2005 [jo]
  
  if (! mydate)
     { SHOWERROR("Bad mem", 1);
       return;
     };
  * mydate = 0x0; 

  /* display the present week commencing date */
  DateToDOWDMFormat((Byte) dayDate->month, (Byte) dayDate->day,
                    (Word) dayDate->year + NINETEENTWENTY,
                    (TSAppPrefs.prefFlags & TSPrefShortDateFlag) ? TSSysDateFormat : TSSysLongDateFormat,
                    mydate);

  /* draw the date string in bold font */
  formPtr = FrmGetActiveForm();
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, labelID_tsDate));
  prevFont = FntSetFont(boldFont);
  length = StrLen(mydate);


///  WinDrawChars(mydate, length, controlPtr->bounds.topLeft.x, controlPtr->bounds.topLeft.y);
  /// [jo] 12-8-2005: We DEFINITELY had a problem here, perhaps related to direct
  /// access to the data structure. Bugger! hack is jox, joy:
  FrmGetObjectPosition(formPtr, FrmGetObjectIndex(formPtr, labelID_tsDate), &jox, &joy);
  WinDrawChars(mydate, length, jox, joy);
  
  
  /* erase the rest of the line */
///  rect.topLeft.x = controlPtr->bounds.topLeft.x + FntCharsWidth(mydate, length);
///  rect.topLeft.y = controlPtr->bounds.topLeft.y;
  // [jo] as above: hack:
  rect.topLeft.x = jox + FntCharsWidth(mydate, length);
  rect.topLeft.y = joy;
  rect.extent.x = 150 - rect.topLeft.x;
  rect.extent.y = 11;
  WinEraseRectangle(&rect, 0);
  FntSetFont(prevFont);

  e_Delete(mydate); 
  // have completely removed datestr  
}
// -------------------------------------------------------------- //



/******************************************************************
 * Application utility functions.
 ******************************************************************/

/******************************************************************
 * Update the form's on-screen scroll up/down buttons.
 ******************************************************************/
void
ScrollButtonUpdate(FormPtr formPtr, UInt topRow, UInt tableRows, UInt totalRows)
{
  ControlPtr controlPtr;

  /* update scroll up button */
  if (topRow > 0)
    {
      /* enable scroll up if req'd */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsTableUpGrey));
      if (controlPtr->attr.visible)
        {
          CtlHideControl(controlPtr);   /* hide the greyed out button */
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsTableUp));
          CtlShowControl(controlPtr);   /* show the usable button */
        }
    }
  else
    {
      /* disable scroll up */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsTableUp));
      CtlHideControl(controlPtr);       /* hide the usable button */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsTableUpGrey));
      CtlShowControl(controlPtr);       /* show the greyed out button */
    }
  /* update scroll down button */
  if (topRow + tableRows < totalRows)
    {
      /* enable scroll down if req'd */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsTableDownGrey));
      if (controlPtr->attr.visible)
        {
          CtlHideControl(controlPtr);   /* hide the greyed out button */
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsTableDown));
          CtlShowControl(controlPtr);   /* show the usable button */
        }
    }
  else
    {
      /* disable scroll down */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsTableDown));
      CtlHideControl(controlPtr);       /* hide the usable button */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, buttonID_tsTableDownGrey));
      CtlShowControl(controlPtr);       /* show the greyed out button */
    }
}

/******************************************************************
 *
 ******************************************************************/
void
TableScroll(Int * topRow, Int tableRows, Int totalRows, Byte dir)
{
  /* Update table top row according to direction */
  if (dir == pageUpChr)
    {
      (*topRow) = (*topRow) - (tableRows - 1);
    }
  else
    {
      (*topRow) = (*topRow) + (tableRows - 1);
    }
  /* Bounds check updated table top row */
  if ((*topRow) + tableRows >= totalRows)
    {
      (*topRow) = totalRows - tableRows;
    }
  if ((*topRow) < 0)
    {
      (*topRow) = 0;
    }
}

/*** End of section! ***/




// ============================================================================ //
// Category:
// #include "Category.c"

/******************************************************************
 * Application category functions.
 ******************************************************************/

/******************************************************************
 * Attempt to delete an existing entry string from the category at
 * the specified point. All transTable entries that correspond to
 * the entry that is deleted will be remapped to 255. Caller is
 * expected to deal with transTable fixup after calling function.
 * Returns the new handle to the category after it has been resized
 * by deleting category.
 ******************************************************************/
VoidHand
CatDeleteEntry(UInt catRecIdx, Byte numEntries, UInt catDelIdx)
{
  VoidHand catHand;
  TSCatRecType *catPtr;
  Byte delMarker;
  Byte trans;
  Byte idx;

  catHand = DmGetRecord(TSDatabase, catRecIdx);
  catPtr = MemHandleLock(catHand);
  /* shuffle the remaining entries down over the deleted entry */
  for (idx = catDelIdx; idx < numEntries - 1; idx += 1)
    {
      DmStrCopy(catPtr, (VoidPtr) (catPtr->cats[idx]) - (VoidPtr) (catPtr), catPtr->cats[idx + 1]);
    }
  /* find the translation table entries for the category we've just deleted */
  delMarker = 255;
  for (idx = 1; idx < TSMaxCatEntries; idx += 1)
    {
      if (catPtr->transTable[idx] == catDelIdx)
        {
          /* mark the deleted translation table index as deleted and let the caller deal with it.
           * note: we don't break out of the loop because multiple transTable entries may point
           * to a single actual entry (which may be the one we've just deleted). */
          DmWrite(catPtr, sizeof(Byte) * idx, &delMarker, sizeof(Byte));
        }
    }
  /* fix up the translation table entries after the deletion point (by subtracting 1!) */
  for (idx = 1; idx < TSMaxCatEntries; idx += 1)
    {
      trans = catPtr->transTable[idx];
      if ((trans >= catDelIdx) && (trans < TSMaxCatEntries))
        {
          trans -= 1;
          DmWrite(catPtr, (VoidPtr) (&(catPtr->transTable[idx])) - (VoidPtr) (catPtr), &trans, sizeof(Byte));
        }
    }
  /* make the category record one entry smaller */
  MemHandleUnlock(catHand);
  DmReleaseRecord(TSDatabase, catRecIdx, true);  /// [c]
  return DmResizeRecord(TSDatabase, catRecIdx, (sizeof(Byte) * TSMaxCatEntries)
                    + (sizeof(Char) * TSMaxCatEntryLen * (numEntries - 1)));
}

/******************************************************************
 * Attempt to insert a new entry string into the category at the
 * appropriate alphabetic point. Returns the index_entry was
 * inserted into, or -1 - index_entry if the entry already exists.
 ******************************************************************/

Int CatInsertEntry(UInt catRecIdx, Byte numEntries, CharPtr newCat)
{
  UInt idx;
  Int result;
  Byte trans;
  Byte newCatIdx;
  VoidHand catHand;
  TSCatRecType *catPtr;
  
  Int16 jolen; // [jo] HERE MAKE TEMP BUFFER TO MINIMISE FUSS. Ok there are cleaner ways.
  Char * jobuf;
  jolen = StrLen(newCat); // get length of string to insert
  jobuf = (Char *) e_New(TSMaxCatEntryLen); // make temp buffer
  /// TSMaxCatEntryLen is specified as 18 in Timesheet.h = aagh!
  xFill(jobuf, TSMaxCatEntryLen, 0x0); // fill with zeroes ?!
  xCopy(jobuf, newCat, jolen);

  /* check there isn't any existing category of the same name, while also searching for the
   * appropriate insertion point for the new entry */
///  catHand = DmQueryRecord(TSDatabase, catRecIdx);
  // [jo] SHOULD WE NOT USE DmGetRecord in the above thus:
    catHand = DmGetRecord(TSDatabase, catRecIdx);
  
  catPtr = MemHandleLock(catHand);
  for (idx = 0, newCatIdx = TSMaxCatEntries; idx < numEntries; idx += 1)
    {
      result = StrCompare(newCat, catPtr->cats[idx]);
      if (result == 0)
        { MemHandleUnlock(catHand);
          DmReleaseRecord(TSDatabase, catRecIdx, false); // [jo] 
          e_Delete(jobuf);
          return -1 - idx;
        }
      else if ((idx > 0) && (idx < numEntries - 1) && (newCatIdx == TSMaxCatEntries) && (result < 0))
        {
          /* have found the appropriate (alphabetic) insertion point for the new category */
          newCatIdx = idx;
          /* note we keep searching for duplicate entries, as last 'Edit...' entry is *not* in alphabetical order */
        }
    }
  /* check we have found insertion point, otherwise default to entry before 'Edit...' */
  if (newCatIdx == TSMaxCatEntries)
    {
      newCatIdx = numEntries - 1;
    }
  MemHandleUnlock(catHand);
  /* make room for insertion of new category string */
  catHand = DmResizeRecord(TSDatabase, catRecIdx, (sizeof(Byte) * TSMaxCatEntries)
                    + (sizeof(Char) * TSMaxCatEntryLen * (numEntries + 1)));
  catPtr = MemHandleLock(catHand);
  for (idx = numEntries; idx > newCatIdx; idx -= 1)
    {
      DmStrCopy(catPtr, (VoidPtr) (catPtr->cats[idx]) - (VoidPtr) (catPtr), catPtr->cats[idx - 1]);
    }
  /* save the field text in the free category */
  DmWrite( catPtr, 
           (VoidPtr) (&(catPtr->cats[newCatIdx])) - (VoidPtr) (catPtr), 
           jobuf, 
           sizeof(Byte) * TSMaxCatEntryLen);
  /// [jo] 13-8-2005 error occurs here = buffer overwrite. What is too small?
  // I think what is happening is that pgm relied on picking up crap from after
  // end of string as 'filler' but PalmOS no longer permits this (wisely).
  /// DmWrite writes to locked record, at offset, from source, for n bytes.
  e_Delete(jobuf);
  
  /* fix up the translation table entries after the insertion point (by adding 1!) */
  for (idx = 1; idx < TSMaxCatEntries; idx += 1)
    {
      trans = catPtr->transTable[idx];
      if ((trans >= newCatIdx) && (trans < TSMaxCatEntries))
        {
          trans += 1;
          DmWrite(catPtr, (VoidPtr) (&(catPtr->transTable[idx])) - (VoidPtr) (catPtr), &trans, sizeof(Byte));
        }
    }
  MemHandleUnlock(catHand);
  DmReleaseRecord(TSDatabase, catRecIdx, true);
  /// [jo] THERE WAS AN ERROR HERE: record wasn't active as used DmQueryRecord above !
  return newCatIdx;
}

/******************************************************************
 * Utility function that returns the index of the entry in the
 * transTable that contains the actual idx passed to the function.
 * Used to produce client/project/task transtable indexes from the
 * actual user GUI list selections.
 ******************************************************************/
Word
CatReverseLookup(Word idx, BytePtr transTable, UInt numEntries)
{
  UInt tblIdx = 1;

  /* quick check to see if we should default to unspecified or 'none' entry */
  if (idx <= 0 || idx >= numEntries)
    {
      /* 'none' entry will always be at index 0 of the translation table */
      return 0;
    }
  /* search the translation table for the table index containing the supplied index! */
  for (; tblIdx < numEntries && transTable[tblIdx] != idx; tblIdx += 1)
    {
      /* this body left intentionally empty */
    }
  return tblIdx;
}

/******************************************************************
 * Build an array of pointers to the strings in the specified category
 * (in translation table order). Typically called before
 * the pointer array is used to initialise a GUI list control.
 * Caller is responsible for freeing returned dynamically allocated pointer.
 *
 * [jo] memory allocation needs corresponding freeing, ULTIMATELY.
 ******************************************************************/
VoidPtr
CatBuildPtrArray(TSCatRecType * catRec, Byte numEntries)
{
  CharPtr *ptr;
  UInt idx;

  /* dynamically allocate space for the pointer array */
  ptr = MemPtrNew(sizeof(CharPtr) * numEntries);
  if (!ptr)
    {
      /* low on memory! should check or halt program! */
      return NULL;
    }
  /* build a pointer array to pass to LstSetListChoices (is there an LstAddChoice function?) */
  for (idx = 0; idx < numEntries; idx += 1)
    {
      ptr[idx] = catRec->cats[idx];
    }
  return ptr;
}

/******************************************************************
 * Utility function, set up a resource list record for the supplied
 * list. There's three lists that are important to Timesheet, the
 * Client list, the Project list and the Task list.
 ******************************************************************/
Err
InitCatResource(UIntPtr resIdP, Int initStringID, UInt numEntries)
{
  Byte idx;
  VoidHand hand;
  VoidHand resHand;
  Ptr catPtr;

  /* create and initialise a new list record to hold the supplied client, project or task list.
   * Note we only allocate enough space to store the initial default entries */
  hand = DmNewRecord(TSDatabase, resIdP, sizeof(Byte) * TSMaxCatEntries
                     + (sizeof(Char) * TSMaxCatEntryLen * numEntries));
  if (!hand)
    {
      /* failed to create record */
      return appErrorClass;
    }
  catPtr = MemHandleLock(hand);
  /* initialise the translation table in the record */
  DmSet(catPtr, 0, sizeof(Byte) * TSMaxCatEntries, TSMaxCatEntries);
  for (idx = 0; idx < numEntries - 1; idx++)
    {
      DmWrite(catPtr, sizeof(Byte) * idx, &idx, sizeof(Byte));
    }
  /* last transTable entry *always* points to 'Edit...' category name */
  DmWrite(catPtr, sizeof(Byte) * (TSMaxCatEntries - 1), &idx, sizeof(Byte));
  /* write the default category names */
  resHand = DmGet1Resource('tSTR', initStringID);
  DmWrite(catPtr, sizeof(Byte) * TSMaxCatEntries, MemHandleLock(resHand),
          sizeof(Char) * TSMaxCatEntryLen * numEntries);
  MemHandleUnlock(resHand);
  DmReleaseResource(resHand);
  /* have created another resource record */
  MemHandleUnlock(hand);
  DmReleaseRecord(TSDatabase, *resIdP, true);
  (*resIdP) += 1;
  return 0;
}

/*** End of section! ***/






// ============================================================================ //
// WeeklyForm:
// #include "WeeklyForm.c"
   
/******************************************************************
 *
 ******************************************************************/
Boolean
WeeklyFormEventHandler(EventPtr event)
{
  SWord day;
  SWord month;
  SWord year;
  VoidHand hand;
  FormPtr formPtr;
  Boolean handled = false;

  if (event->eType == ctlSelectEvent)
    {
      /* A control button was clicked */
      if (event->data.ctlEnter.controlID == buttonID_tsDayViewButton)
        {
          /* Change to the main daily view */
          FrmGotoForm(formID_tsMain);
          handled = true;
        }
      else if (event->data.ctlEnter.controlID == buttonID_tsNextWeek)
        {
          /* Select the next week */
          WeeklyFormChangeDate(7);
          handled = true;
        }
      else if (event->data.ctlEnter.controlID == buttonID_tsPrevWeek)
        {
          /* Select the previous week */
          WeeklyFormChangeDate(-7);
          handled = true;
        }
      else if (event->data.ctlEnter.controlID == buttonID_tsGoto)
        {
          /* Jump to user specified date */
          day = TSPresentDayDate.day;
          month = TSPresentDayDate.month;
          year = TSPresentDayDate.year + NINETEENTWENTY;
          hand = DmGet1Resource('tSTR', stringID_tsSelectDate);
          if (SelectDay(selectDayByWeek, &month, &day, &year, (CharPtr) MemHandleLock(hand)))
            {
              /* User has entered date to go to, save the new date */
              TSPresentDayDate.day = day;
              TSPresentDayDate.month = month;
              TSPresentDayDate.year = year - NINETEENTWENTY;
              /* Find the day record for today (if any) */
              FindDayRec(&TSPresentDayDate, &TSPresentDayRecIdx, &TSPresentDayRecExists);
              /* Reinitialise the weekly form (this is a cheap way of getting display updated) */
              WeeklyFormInit();
              /* Update the weekly form GUI */
              WeeklyFormUpdateWeek();
              FormUpdateDay(&TSPresentSummaryDate);
              WeeklyFormTableDraw();
            }
          MemHandleUnlock(hand);
          DmReleaseResource(hand);
          handled = true;
        }
    }
  else if (event->eType == ctlRepeatEvent)
    {
      if (event->data.ctlSelect.controlID == buttonID_tsTableUp)
        {
          /* Scroll table up */
          WeeklyFormTableScroll(pageUpChr);
          WeeklyFormTableDraw();
        }
      else if (event->data.ctlSelect.controlID == buttonID_tsTableDown)
        {
          /* Scroll table down */
          WeeklyFormTableScroll(pageDownChr);
          WeeklyFormTableDraw();
        }
    }
  else if (event->eType == menuEvent)
    {
      /* A menu item has been selected */
      if (event->data.menu.itemID == menuitemID_tsGotoEarliest)
        {
          /* User wants to go to earliest day in database */
          WeeklyFormGotoEarliestWeek();
        }
      else if (event->data.menu.itemID == menuitemID_tsGotoLatest)
        {
          /* User wants to go to latest day in database */
          WeeklyFormGotoLatestWeek();
        }
      else if (event->data.menu.itemID == menuitemID_tsEraseWeek)
        {
          hand = DmGet1Resource('tSTR', stringID_tsWeek);
          if (GetNumSummaryRecords() > 1)
            {
              /* User wants to erase the current summary week */
              if (FrmCustomAlert(alertID_tsConfirmDayWeekDel, 
                  (CharPtr)MemHandleLock(hand), NULL, NULL) == 0)
              {
                /* User confirms delete, delete all entries for this week */
                WeeklyFormEraseWeek();
              }
            }
          else
            {
              /* Nothing in this week to erase */
              FrmCustomAlert(alertID_tsEmptyDayWeek, (CharPtr)MemHandleLock(hand), NULL, NULL);
            }
          MemHandleUnlock(hand);
          DmReleaseResource(hand);        
        }
      else if (event->data.menu.itemID == menuitemID_tsPref)
        {
          /* display the prefs form */
          FrmPopupForm(formID_tsPref);
          handled = true;
        }
      else if (event->data.menu.itemID == menuitemID_tsAbout)
        {
          /* display the About form */
          FrmPopupForm(formID_tsAbout);
          /* FrmHelp(helpID_tsAbout); */
          handled = true;
        }
    }
  else if (event->eType == keyDownEvent)
    {
      /* Hardware key has been pressed, is it one app should respond to? */
      if (event->data.keyDown.chr == pageUpChr)
        {
          /* Command key up, goto previous week */
          WeeklyFormChangeDate(-7);
          handled = true;
        }
      else if (event->data.keyDown.chr == pageDownChr)
        {
          /* Command key down, goto next week */
          WeeklyFormChangeDate(7);
          handled = true;
        }
    }
  else if (event->eType == tblEnterEvent)
    {
      /* Do nothing, stops UI from inverting tapped columns */
      handled = true;
    }
  else if (event->eType == frmOpenEvent)
    {
      /* The Weekly form has been opened */
      formPtr = FrmGetActiveForm();
      FrmDrawForm(formPtr);
      /* Initialise the weekly form */
      WeeklyFormInit();
      /* Draw the weekly form */
      WeeklyFormUpdateWeek();
      FormUpdateDay(&TSPresentSummaryDate);
      WeeklyFormTableDraw();
      handled = true;
    }
  return handled;
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormUpdateWeek(void)
{
  FormPtr formPtr;
  ControlPtr controlPtr;
  Word weekNum;
  /// Char weekTxt[4 + 1];
  Char * weekTxt; ///
  FontID prevFont;
  RectangleType controlRect;
  Coord jox;
  Coord joy; // my little hack. 
  
  weekTxt = e_New(5); ///

  formPtr = FrmGetActiveForm();
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, labelID_tsWeek));
  /* Get the week number of the current week (thanks to Palm Computing) */
  weekNum = GetWeekNumber(TSPresentSummaryDate.month, TSPresentSummaryDate.day, TSPresentSummaryDate.year + NINETEENTWENTY);
  /* Erase the old week number */
  MemMove(&controlRect, &(controlPtr->bounds), sizeof(RectangleType));
  /* Manually set extents based on font size */
  controlRect.extent.x = 10;
  controlRect.extent.y = 11;
  WinEraseRectangle(&controlRect, 0);
  /* Display the week number */
  StrIToA(weekTxt, weekNum);
  prevFont = FntSetFont(stdFont);

  FrmGetObjectPosition(formPtr, FrmGetObjectIndex(formPtr, labelID_tsWeek), &jox, &joy);
  WinDrawChars(weekTxt, StrLen(weekTxt), jox, joy);
  FntSetFont(prevFont);
  
  e_Delete(weekTxt); /// 
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormInit(void)
{
  DateType startDate;
  DateType endDate;
  Int weekDay;

  /* Find the date for the start of this week */
  weekDay = DayOfWeek(TSPresentDayDate.month, TSPresentDayDate.day, TSPresentDayDate.year + NINETEENTWENTY);
  MemMove(&startDate, &TSPresentDayDate, sizeof(DateType));
  /* Adjust to week start depending on palm system settings */
  if (TSSysWeekStartDay == 0)
  {
        /* Week starts on Sunday */
    DateAdjust(&startDate, -weekDay);   
  } 
  else
  {
        /* Week starts on Monday */
    DateAdjust(&startDate, -( weekDay == 0 ? 6 : weekDay - 1));
  }
  /* Set global summary date for week start */
  MemMove(&TSPresentSummaryDate, &startDate, sizeof(DateType));
  /* Find the date for the end of this week */
  MemMove(&endDate, &startDate, sizeof(DateType));
  DateAdjust(&endDate, 6);
  /* Build a summary database */
  BuildSummaryDatabase(&startDate, &endDate);
  /* Select the week view button */
  FrmSetControlGroupSelection(FrmGetActiveForm(), (UInt8) groupID_tsViewGroup, (UInt16) buttonID_tsWeekViewButton); ///
  /* Initialise the table globals */
  TSTableTopEntry = 0;
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormGotoEarliestWeek(void)
{
  /* Find the earliest day rec (if one exists) */
  if (FindEarliestDayRec())
    {
      /* Find the day record for today (if any) */
      FindDayRec(&TSPresentDayDate, &TSPresentDayRecIdx, &TSPresentDayRecExists);
      /* Reinitialise the weekly form (this is a cheap way of getting display updated) */
      WeeklyFormInit();
      /* Update the weekly form GUI */
      WeeklyFormUpdateWeek();
      FormUpdateDay(&TSPresentSummaryDate);
      WeeklyFormTableDraw();
    }
  else
    {
      /* no earlier day to go to, alert user */
      FrmAlert(alertID_tsEmptyDB);
    }
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormGotoLatestWeek(void)
{
  /* Find the earliest day rec (if one exists) */
  if (FindLatestDayRec())
    {
      /* Find the day record for today (if any) */
      FindDayRec(&TSPresentDayDate, &TSPresentDayRecIdx, &TSPresentDayRecExists);
      /* Reinitialise the weekly form (this is a cheap way of getting display updated) */
      WeeklyFormInit();
      /* Update the weekly form GUI */
      WeeklyFormUpdateWeek();
      FormUpdateDay(&TSPresentSummaryDate);
      WeeklyFormTableDraw();
    }
  else
    {
      /* no earlier day to go to, alert user */
      FrmAlert(alertID_tsEmptyDB);
    }
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormChangeDate(Int deltaDays)
{
  /* Modify the global dates accordingly */
  DateAdjust(&TSPresentDayDate, deltaDays);
  /* Find the day record for today (if any) */
  FindDayRec(&TSPresentDayDate, &TSPresentDayRecIdx, &TSPresentDayRecExists);
  /* Reinitialise the weekly form (this is a cheap way of getting display updated) */
  WeeklyFormInit();
  /* Update the weekly form GUI */
  WeeklyFormUpdateWeek(); 
  FormUpdateDay(&TSPresentSummaryDate);
  WeeklyFormTableDraw();
}

/******************************************************************
 * Erase all entries in the current week.
 ******************************************************************/
void WeeklyFormEraseWeek(void)
{
  DateType date;
  LocalID summaryID;
  Int summaryCardNum;  
  DmOpenRef summaryDB;  

  /* Find the week end date */
  MemMove(&date, &TSPresentSummaryDate, sizeof(DateType));
  DateAdjust(&date, 6);
  EraseDatabaseRange(&TSPresentSummaryDate, &date);
  /* Check if there's an existing summary database */
  summaryDB = DmOpenDatabaseByTypeCreator(TSSummaryDBType, TSAppID, dmModeReadWrite);
  if (summaryDB)
    {
      /* Get locaID, cardID etc. so we can delete database */
      DmOpenDatabaseInfo(summaryDB, &summaryID, NULL, NULL, &summaryCardNum, NULL);
      /* Close and delete existing summary database */
      DmCloseDatabase(summaryDB);
      DmDeleteDatabase(summaryCardNum, summaryID);
    }
  /* Nothing to display, move table to top row */
  TSTableTopEntry = 0;
  /* Redraw the main table */
  WeeklyFormTableDraw();
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormTableDrawCell(VoidPtr tablePtr, Word row, Word column, RectanglePtr bounds)
{
  /// Char txt[25 + 1];
  Char * txt;
  UInt idx;
  SWord txtWidth;  
  SWord txtLen;
  Boolean txtFitsInWidth;
  FontID prevFont;
  VoidHand listHand;
  TSCatRecType *listPtr;
  RectangleType rowRect =
  {
    {0, 0},             /* WARNING: note these are hardcoded bounds! */
    {159, 11}
  };                    

  /* Start fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_PROLOGUE;
  txt = e_New(26); ///

  /* Get the entry string index from the item */
  idx = TblGetItemInt(tablePtr, row, column);
  /* Format the column appropriately (either text or HOURmin) */
  if (column == 0)
    {
      /* Erase the entire row once before we draw the first column. This reduces screen flicker, and is more efficient
       * than erasing each individual cell before re-filling it. */
      rowRect.topLeft.x = bounds->topLeft.x;
      rowRect.topLeft.y = bounds->topLeft.y;
      WinEraseRectangle(&rowRect, 0);
      /* Draw the hour in the large font */
      prevFont = FntSetFont(boldFont);
      /* Check we have some hours to show */
      if (idx & 0x0FFF)
        {
          StrIToA(txt, idx & 0x0FFF);
          txtLen = StrLen(txt);
          WinDrawChars(txt, txtLen, bounds->topLeft.x + (18 - (6 * txtLen)), bounds->topLeft.y);
        }
      /* Draw the minutes in the normal font */
      txt[0] = 48 + ((idx & 0xE000) >> 13);
      txt[1] = 48 + (5 * ((idx & 0x1000) >> 12));
      txt[2] = NULL;
      FntSetFont(stdFont);
      WinDrawChars(txt, 2, bounds->topLeft.x + 18, bounds->topLeft.y);
    }
  else
    {
      /* Get the appropriate category record */
      listHand = DmQueryRecord(TSDatabase, column);
      listPtr = MemHandleLock(listHand);

      /* Get the string to put in the cell */
      StrCopy(txt, listPtr->cats[idx]);
      MemHandleUnlock(listHand);

      /* Trim the client string to fit within the cell */
      prevFont = FntSetFont(stdFont);

      /* Draw the cell contents */
      txtWidth = bounds->extent.x;
      txtLen = StrLen( txt );
      FntCharsInWidth( txt, &txtWidth, &txtLen, &txtFitsInWidth );
      txt[ txtLen ] = NULL;
      WinDrawChars(txt, txtLen, bounds->topLeft.x, bounds->topLeft.y);
    }
  /* Restore the previous font */
  FntSetFont(prevFont);

  e_Delete(txt);
  /* End fixup from the GNU Palmpilot SDK FAQ so callback function can access global variables */
  CALLBACK_EPILOGUE;
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormTableDraw(void)
{
  FormPtr formPtr;
  TablePtr tablePtr;
  ControlPtr controlPtr;
  DmOpenRef summaryDB;
  Int numSummaryRecs;
  Int recIdx;
  UInt row;
  UInt col;
  VoidHand recHand;
  TSSummaryRecType *recPtr;
  TSSummaryTotalRecType *summaryTotalPtr;
  UInt totalHours;
  Byte totalMins;
  UInt totalChargeableHours;
  Byte totalChargeableMins;  
  /// Char txt[3]; ///
  Char * txt;
  Byte txtLen;
  Coord jox;
  Coord joy; // my little hack. 
  txt = e_New(3); ///

  /* Get a pointer to the weekly form as it may not be the visible one at the moment */
  formPtr = FrmGetFormPtr(formID_tsWeekly);
  /* Get the table pointer */
  tablePtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, tableID_tsMain));
  /* Check if there's an existing summary database to fill table with */
  numSummaryRecs = GetNumSummaryRecords();
  /* Open summary database, this may return a NULL reference if no summary database exists */
  summaryDB = DmOpenDatabaseByTypeCreator(TSSummaryDBType, TSAppID, dmModeReadWrite);
  /* Skip to first record that currently appears at the top of the table */
  recIdx = TSTableTopEntry;
  for (row = 0; row < TSSummaryFormTableRows; row++, recIdx++)
    {
      if (recIdx < numSummaryRecs - 1 /* last record contains total hours/mins */ )
        {
          /* Have a record to display */
          recHand = DmQueryRecord(summaryDB, recIdx);
          recPtr = MemHandleLock(recHand);

          /* Set summary hours and minutes */
          TblSetItemStyle(tablePtr, row, 0, customTableItem);
          TblSetItemInt(tablePtr, row, 0, recPtr->totalDuration);

          /* Set client column */
          TblSetItemStyle(tablePtr, row, 1, customTableItem);
          /* Translate the client entry index into the actual client index using the client trans table */
          TblSetItemInt(tablePtr, row, 1, recPtr->catIdx[0]);

          /* Set project column */
          TblSetItemStyle(tablePtr, row, 2, customTableItem);
          /* Translate the client entry index into the actual client index using the client trans table */
          TblSetItemInt(tablePtr, row, 2, recPtr->catIdx[1]);

          /* Set task column */
          TblSetItemStyle(tablePtr, row, 3, customTableItem);
          /* Translate the client entry index into the actual client index using the client trans table */
          TblSetItemInt(tablePtr, row, 3, recPtr->catIdx[2]);

          /* Finished setting up columns */
          MemHandleUnlock(recHand);
          /* Data for this row, make it usable and selectable */
          TblSetRowUsable(tablePtr, row, true);
          /* Mark the row invalid, so it re-draws */
          TblMarkRowInvalid(tablePtr, row);
        }
      else
        {
          /* No data for this row, make it unusable */
          TblSetRowUsable(tablePtr, row, false);
        }
    }
  /* Display last record totals */
  if (summaryDB)
    {
      recHand = DmQueryRecord(summaryDB, numSummaryRecs - 1);
      summaryTotalPtr = MemHandleLock(recHand);
      totalHours = summaryTotalPtr->totalHours;
      totalMins = summaryTotalPtr->totalMins;
      totalChargeableHours = summaryTotalPtr->totalChargeableHours;
      totalChargeableMins = summaryTotalPtr->totalChargeableMins;  
      MemHandleUnlock(recHand);
      /* Finished with summary database */
      DmCloseDatabase(summaryDB);
    } 
  else    
    {
      totalHours = 0;
      totalMins = 0;
      totalChargeableHours = 0;
      totalChargeableMins = 0;
    }
  /* Make all columns in the table usable */
  for (col = 0; col < TSMainFormTableCols; col++)
    {
      TblSetColumnUsable(tablePtr, col, true);
    }
  /* Set column spacing */
  TblSetColumnSpacing(tablePtr, 0, 3);
  TblSetColumnSpacing(tablePtr, 1, 3);
  TblSetColumnSpacing(tablePtr, 2, 3);
  TblSetColumnSpacing(tablePtr, 3, 0);
  /* Set the custom drawn columns */
  TblSetCustomDrawProcedure(tablePtr, 0, (TableDrawItemFuncPtr)WeeklyFormTableDrawCell);      /* Hour / mins column */ ///
  TblSetCustomDrawProcedure(tablePtr, 1, (TableDrawItemFuncPtr)WeeklyFormTableDrawCell);      /* Client column */ ///
  TblSetCustomDrawProcedure(tablePtr, 2, (TableDrawItemFuncPtr)WeeklyFormTableDrawCell);      /* Project column */ ///
  TblSetCustomDrawProcedure(tablePtr, 3, (TableDrawItemFuncPtr)WeeklyFormTableDrawCell);      /* Task column */ ///
  /* Finally redraw the table */
  TblDrawTable(tablePtr);
  if (numSummaryRecs - 1 > 0)
    {
      /* Draw a little line to indicate it's a total */
      controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, labelID_tsTotal));

  FrmGetObjectPosition(formPtr, FrmGetObjectIndex(formPtr, labelID_tsTotal), &jox, &joy);
      col = jox;
      row = joy;
      WinDrawLine(col, row - 2, col + 28, row - 2);
      if (totalHours > 0)
        {
          /* Draw the total hours */
          StrIToA(txt, totalHours);
          txtLen = StrLen(txt);
          FntSetFont(boldFont);
          WinDrawChars(txt, txtLen, col + (18 - (6 * txtLen)), row);
        }
      /* Always draw the minutes in the normal font */
      FntSetFont(stdFont);
      txt[0] = 48 + ((totalMins & 0x0E) >> 1);
      txt[1] = 48 + (5 * (totalMins & 0x01));
      txt[2] = NULL;
      WinDrawChars(txt, 2, col + 18, row);
          
      /* Draw the chargeable hour and minute totals if req'd. 
       * Note, this preference is 'inverted', 1 = Off, 0 = On so it's on by default */
      if (!(TSAppPrefs.prefFlags & TSPrefChargeTotalsFlag))
        {
          /* Draw a little line to indicate it's a total */
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, labelID_tsTotal));
          FrmGetObjectPosition(formPtr, FrmGetObjectIndex(formPtr, labelID_tsTotal), &jox, &joy);
//          col = controlPtr->bounds.topLeft.x + 38;
//          row = controlPtr->bounds.topLeft.y;
          col = jox + 38;
          row = joy;
          WinDrawLine(col, row - 2, col + 28, row - 2);
          if (totalChargeableHours > 0)
            {
              /* Draw the total hours */
              StrIToA(txt, totalChargeableHours);
              txtLen = StrLen(txt);
              FntSetFont(boldFont);
              WinDrawChars(txt, txtLen, col + (18 - (6 * txtLen)), row);
            }
          /* Always draw the minutes in the normal font */
          FntSetFont(stdFont);
          txt[0] = 48 + ((totalChargeableMins & 0x0E) >> 1);
          txt[1] = 48 + (5 * (totalChargeableMins & 0x01));
          txt[2] = NULL;
          WinDrawChars(txt, 2, col + 18, row);
        }
    }
  /* Update the on-screen scroll buttons */
  WeeklyFormScrollButtonUpdate(formPtr);
  e_Delete(txt);
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormTableScroll(Byte dir)
{
  Int numSummaryRecs;

  /* Check if there's an existing summary database */
  numSummaryRecs = GetNumSummaryRecords();
  if (numSummaryRecs > 0)  
    {
      numSummaryRecs --; /* Ignore last totals record */
    }
  /* Update the scroll buttons */
  TableScroll(&TSTableTopEntry, TSSummaryFormTableRows, numSummaryRecs, dir);
}

/******************************************************************
 *
 ******************************************************************/
void
WeeklyFormScrollButtonUpdate(FormPtr formPtr)
{
  Int numSummaryRecs;

  /* Check if there's an existing summary database */
  numSummaryRecs = GetNumSummaryRecords();
  if (numSummaryRecs > 0)  
    {
      numSummaryRecs --; /* Ignore last totals record */
    }
  /* Update the scroll buttons */
  ScrollButtonUpdate(formPtr, TSTableTopEntry, TSSummaryFormTableRows, numSummaryRecs);
}

/***********************************************************************
 *
 * FUNCTION:    FirstDayOfYear
 *
 * DESCRIPTION: Return the number of day from 1/1/1904 of the first day 
 *              of the year passed.
 *
 *                                       The first day of the year is always a Monday.  The rule 
 *                                       for determining the first day of the year is:
 *                              
 *                                       New Years Day  First Day of the Year
 *                                       ------------   ---------------------
 *                                       Monday                 Monday Jan 1
 *                                       Tuesday                        Monday Dec 31
 *                                       Wednesday              Monday Dec 30
 *                                       Thursday               Monday Dec 29
 *                                       Friday                 Monday Jan 4 
 *                                       Saturday               Monday Jan 3 
 *                                       Sunday                 Monday Jan 2
 *
 * PARAMETERS:   year  - year (1904-2031)
 *
 * RETURNED:     number of days since 1/1/1904
 *
 * REVISION HISTORY:
 *                      Name    Date            Description
 *                      ----    ----            -----------
 *                      art     4/4/96  Initial Revision
 *
 ***********************************************************************/
ULong
FirstDayOfYear(Word year)
{
  ULong days;
  Word dayOfWeek;
  DateType date;

  /* Get days to January 1st of the year passed. */
  date.day = 1;
  date.month = 1;
  date.year = year - firstYear;
  days = DateToDays(date);

  dayOfWeek = DayOfWeek(1, 1, year);

  /* Move to monday. */
  days++;
  days -= dayOfWeek;


  if (dayOfWeek >= friday)
    days += daysInWeek;

  return (days);
}


/***********************************************************************
 *
 * FUNCTION:    GetWeekNumber
 *
 * DESCRIPTION: Calculate the week number of the specified date.
 *
 * PARAMETERS:   month - month (1-12)
 *              day   - day (1-31)
 *              year  - year (1904-2031)
 *
 * RETURNED:     week number (1-53)
 *
 * REVISION HISTORY:
 *                      Name    Date            Description
 *                      ----    ----            -----------
 *                      art     4/4/96  Initial Revision
 *
 ***********************************************************************/
Word
GetWeekNumber(Word month, Word day, Word year)
{
  Word dow;
  Word week;
  ULong days;
  ULong firstOfYear;
  ULong firstOfNextYear;
  DateType date;

  /* Calculate the julian date of Monday in the same week as the specified date. */
  date.day = day;
  date.month = month;
  date.year = year - firstYear;
  days = DateToDays(date);

  /* Adjust the day of the week by the preference setting for the first day of the week. */
  dow = (DayOfWeek(month, day, year) - TSSysWeekStartDay + daysInWeek)
    % daysInWeek;

  if (monday < TSSysWeekStartDay)
    days -= (Short) dow - (monday + daysInWeek - TSSysWeekStartDay);
  else
    days -= (Short) dow - (monday - (Short) TSSysWeekStartDay);


  firstOfYear = FirstDayOfYear(year);

  if (days < firstOfYear)
    {
      /* The date passed is in a week that is part of the prior year, so get the start of the prior year. */
      if (year > firstYear)
        firstOfYear = FirstDayOfYear(--year);
    }
  else
    {
      /* Make sure the date passed is not in a week that in part of next year. */
      if (year < lastYear)
        {
          firstOfNextYear = FirstDayOfYear(year + 1);
          if (days == firstOfNextYear)
            firstOfYear = firstOfNextYear;
        }
    }

  week = ((Short) (days - firstOfYear)) / daysInWeek + 1;       /* one base */

  return (week);
}

/*** End of section! ***/





// ============================================================================ //
// EditCatForm:
// #include "EditCatForm.c"

/******************************************************************
 * Close the EditCat form, releasing any dynamically allocated memory
 * [jo] needs work!
 ******************************************************************/
void
EditCatFormClose(FormPtr formPtr)
{
  ListPtr listPtr;

  listPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsEdit));
  /* this is an extremely dodgy bit of coding that matches the other dodgy bit of coding
   * where we initialise the list with the second (not the first) list pointer */
  MemPtrFree(listPtr->itemsText - 1);
}

/******************************************************************
 * Event handler for the Edit List form.
 ******************************************************************/
Boolean
EditCatFormEventHandler(EventPtr event)
{
  Word result;
  Word selected;
  FormPtr formPtr;
  ControlPtr controlPtr;
  ControlPtr controlPtr2;
  Boolean handled = false;

  if (event->eType == ctlSelectEvent)
    {
      if (event->data.ctlEnter.controlID == buttonID_tsOk)
        {
          /* return to the previous form and redraw the form */
          EditCatFormClose(FrmGetActiveForm());
          FrmReturnToForm(formID_tsDetail);
          /* update all the popup triggers as well (as underlying list contents may have changed */
          formPtr = FrmGetActiveForm();
          /* client popup */
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsClient));
          selected = LstGetSelection((ListPtr) controlPtr);
          controlPtr2 = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listpopupID_tsClient));
          CtlSetLabel(controlPtr2, LstGetSelectionText((ListPtr) controlPtr, selected));
          /* project popup */
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsProject));
          selected = LstGetSelection((ListPtr) controlPtr);
          controlPtr2 = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listpopupID_tsProject));
          CtlSetLabel(controlPtr2, LstGetSelectionText((ListPtr) controlPtr, selected));
          /* task popup */
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsTask));
          selected = LstGetSelection((ListPtr) controlPtr);
          controlPtr2 = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listpopupID_tsTask));
          CtlSetLabel(controlPtr2, LstGetSelectionText((ListPtr) controlPtr, selected));
          /* redraw the form */
          FrmDrawForm(formPtr);
          handled = true;
        }
      else if (event->data.ctlEnter.controlID == buttonID_tsNew)
        {
          /* user wants to add a new entry, check if user can add another entry */
          if (TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1] == TSMaxCatEntries)
            {
              /* category is already full */
              FrmAlert(alertID_tsCatFull);
              handled = true;
              return handled;
            }
          /* we can add another entry to the category list */
          TSEditCatEntryIdx = 0;
          FrmPopupForm(formID_tsEditCatEntry);
          handled = true;
        }
      else if (event->data.ctlEnter.controlID == buttonID_tsRename)
        {
          /* user wants to rename existing entry */
          formPtr = FrmGetActiveForm();
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsEdit));
          if ((result = LstGetSelection((ListPtr) controlPtr)) == -1)
            {
              /* user doesn't have a category selected */
              FrmAlert(alertID_tsSelectCat);
              handled = true;
              return handled;
            }
          /* we can renmae an existing entry in the category list */
          TSEditCatEntryIdx = result + 1;       /* add one as entry 0 'none' is not user-editable */
          FrmPopupForm(formID_tsEditCatEntry);
          handled = true;
        }
      else if (event->data.ctlEnter.controlID == buttonID_tsDelete)
        {
          /* user wants to delete existing selected entry */
          formPtr = FrmGetActiveForm();
          controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsEdit));
          if ((result = LstGetSelection((ListPtr) controlPtr)) == -1)
            {
              /* user doesn't have a category selected */
              FrmAlert(alertID_tsSelectCat);
              handled = true;
              return handled;
            }
          /* warn the user and get confirmation of delete */
          if (FrmAlert(alertID_tsConfirmCatDel) == 0)
            {
              /* delete the category user has selected */
              EditCatEntryDeleteEntry(result + 1);      /* add one as 0-'none' entry is not editable or shown to user */
              /* re-initialise this form now that there's one less entry */
              EditCatFormInit(-1, true);
              FrmDrawForm(FrmGetActiveForm());
              /* need to update appropriate list on the 1-down  Detail form */
              if (TSEditCatRecIdx == 1)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListDelete(listID_tsClient, listpopupID_tsClient, 1, TSAppPrefs.numCatEntries[0],
                                            result + 1);
                }
              else if (TSEditCatRecIdx == 2)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListDelete(listID_tsProject, listpopupID_tsProject, 2, TSAppPrefs.numCatEntries[1],
                                            result + 1);
                }
              else if (TSEditCatRecIdx == 3)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListDelete(listID_tsTask, listpopupID_tsTask, 3, TSAppPrefs.numCatEntries[2], result + 1);
                }
            }
          handled = true;
        }
    }
  else if (event->eType == frmOpenEvent)
    {
      /* the EditCat form has been opened */
      EditCatFormInit(-1, false);
      FrmDrawForm(FrmGetActiveForm());
      handled = true;
    }

  return handled;
}

/******************************************************************
 * Initialise the edit list form, which is about to be shown.
 * The single parameter specifies which list entry should be
 * selected after initialisation has completed. Pass -1 for no
 * selection. If freeExisting is set, the existing list entry memory
 * is released before the list is re-built.
 ******************************************************************/
void
EditCatFormInit(Int selection, Boolean freeExisting)
{
  CharPtr *catPtr;
  ListPtr listPtr;
  VoidHand listHand;
  FormPtr formPtr;
  TSCatRecType *listRec;

  /* open the resource record user wants to edit */
  formPtr = FrmGetActiveForm();
  listPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, listID_tsEdit));
  listHand = DmQueryRecord(TSDatabase, TSEditCatRecIdx);
  listRec = MemHandleLock(listHand);
  /* free previously allocated list entries */
  if (freeExisting)
    {
      /* this is an extremely dodgy bit of coding that matches the other dodgy bit of coding
       * where we initialise the list with the second (not the first) list pointer, see below */
      MemPtrFree(listPtr->itemsText - 1);
    }
  /* rebuild the pointer array in case OS has moved records around in memory */
  catPtr = CatBuildPtrArray(listRec, TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1]);
  /* fill it with the supplied entries */
  LstSetListChoices(listPtr, &(catPtr[1]), TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1] - 2);
  MemHandleUnlock(listHand);
  /* set the list's selection */
  LstSetSelection(listPtr, selection);
}

/*** End of section! ***/


// ============================================================================ //
// EditCatEntryForm:
// #include "EditCatEntryForm.c"

/******************************************************************
 * Event handler for the Edit Category form.
 ******************************************************************/
Boolean
EditCatEntryFormEventHandler(EventPtr event)
{
  Boolean handled = false;
  Int result;
  FormPtr formPtr;

  if (event->eType == ctlSelectEvent)
    {
      /* a control button was pressed and released */
      if (event->data.ctlEnter.controlID == buttonID_tsOk)
        {
          /* check if we're create a new category or renaming an existing category */
          if (TSEditCatEntryIdx == 0)
            {
              /* add the new entry to the category */
              if ((result = EditCatEntryAddEntry()) == -1)
                {
                  /* do nothing, user hasn't entered valid new category name */
                  handled = true;
                  return handled;
                }               /* else fall through to code below */
              /* show the 1-down category edit form again */
              FrmReturnToForm(formID_tsEditCat);
              /* and re-initialise it now that there's a new entry */
              EditCatFormInit(result - 1, true);
              FrmDrawForm(FrmGetActiveForm());
              /* also need to update appropriate list on the 2-down Detail form */
              if (TSEditCatRecIdx == 1)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListAdd(listID_tsClient, listpopupID_tsClient, 1,
                                       TSAppPrefs.numCatEntries[0], result);
                }
              else if (TSEditCatRecIdx == 2)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListAdd(listID_tsProject, listpopupID_tsProject, 2,
                                       TSAppPrefs.numCatEntries[1], result);
                }
              else if (TSEditCatRecIdx == 3)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListAdd(listID_tsTask, listpopupID_tsTask, 3,
                                       TSAppPrefs.numCatEntries[2], result);
                }
            }
          else
            {
              /* rename an existing, selected category */
              if ((result = EditCatEntryRenameEntry()) == -1)
                {
                  /* do nothing, user hasn't entered valid category name */
                  handled = true;
                  return handled;
                }
              else if (result == 0)
                {
                  /* user hasn't changed category details, do nothing but return to previous form */
                  FrmReturnToForm(formID_tsEditCat);
                  handled = true;
                  return handled;
                }
              /* show the 1-down category edit form again */
              FrmReturnToForm(formID_tsEditCat);
              /* and re-initialise it now that there's a new entry */
              EditCatFormInit(result - 1, true);
              FrmDrawForm(FrmGetActiveForm());
              /* also need to update appropriate list on the 2-down Detail form */
              if (TSEditCatRecIdx == 1)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListRename(listID_tsClient, listpopupID_tsClient, 1,
                    TSAppPrefs.numCatEntries[0], TSEditCatEntryIdx, result);
                }
              else if (TSEditCatRecIdx == 2)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListRename(listID_tsProject, listpopupID_tsProject, 2,
                    TSAppPrefs.numCatEntries[1], TSEditCatEntryIdx, result);
                }
              else if (TSEditCatRecIdx == 3)
                {
                  /* refill the list and update the current selection */
                  DetailFormPopupListRename(listID_tsTask, listpopupID_tsTask, 3,
                    TSAppPrefs.numCatEntries[2], TSEditCatEntryIdx, result);
                }
            }
          handled = true;
        }
      else if (event->data.ctlEnter.controlID == buttonID_tsCancel)
        {
          /* cancel category form and show the detail form again */
          FrmReturnToForm(formID_tsEditCat);
          handled = true;
        }
    }
  else if (event->eType == frmOpenEvent)
    {
      /* the form has been opened */
      formPtr = FrmGetActiveForm();
      FrmDrawForm(formPtr);
      FrmSetFocus(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsCat));
      handled = true;
    }

  return handled;
}

/******************************************************************
 * Attempt to add a new user-entered entry to the category currently
 * being edited. Returns the entry index of the new entry if added,
 * otherwise returns -1 if the entry wasn't added for some reason.
 ******************************************************************/
Int
EditCatEntryAddEntry(void)
{
  CharPtr newCat;
  Byte idx;
  Int newCatIdx;
  Byte freeTransIdx;
  FormPtr formPtr;
  FieldPtr fieldPtr;
  VoidHand catHand;
  TSCatRecType *catPtr;

  formPtr = FrmGetActiveForm();
  fieldPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsCat));
  /* check the user has actually entered a new category */
  newCat = FldGetTextPtr(fieldPtr);
  if (newCat == NULL || newCat[0] == NULL || IsAllSpace(newCat))
    {
      /* string appears to be empty, alert the user and return */
      FrmAlert(alertID_tsEmptyCat);
      return -1;
    }
  /* attempt to insert the new entry into the entry array alphabetically */
  if ((newCatIdx = CatInsertEntry(TSEditCatRecIdx, TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1], newCat)) < 0)
    {
      /* user attempts to add category that already exists */
      FrmAlert(alertID_tsExistingCat);
      return -1;
    }
  /* now comes the hard part... find a free entry in the category trans table for this new category */
  catHand = DmGetRecord(TSDatabase, TSEditCatRecIdx);
  catPtr = MemHandleLock(catHand);
  do
    {
      for (idx = 0, freeTransIdx = TSMaxCatEntries; idx < TSMaxCatEntries; idx += 1)
        {
          if (catPtr->transTable[idx] == TSMaxCatEntries)
            {
              freeTransIdx = idx;
              break;
            }
        }
      /* do we have a free index, or is a database reorg required? */
      if (freeTransIdx == TSMaxCatEntries)
        {
          /* reorg is required */
          if (!ReorgDatabase(catPtr, TSEditCatRecIdx))
            {
              /* reorg has freed no entries, category really is completely full */
              FrmAlert(alertID_tsCatFull);
              MemHandleUnlock(catHand);
              DmReleaseRecord(TSDatabase, TSEditCatRecIdx, false);
              return -1;
            }
        }
    }
  while (freeTransIdx == TSMaxCatEntries);
  /* have created a new entry, now map the entry into the trans table */
  idx = newCatIdx;              /* type conversion */
  DmWrite(catPtr, (VoidPtr) (&(catPtr->transTable[freeTransIdx])) - (VoidPtr) (catPtr), &idx, sizeof(Byte));
  /* successfully added new category */
  TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1] += 1;
  MemHandleUnlock(catHand);
  DmReleaseRecord(TSDatabase, TSEditCatRecIdx, true);
  return newCatIdx;
}

/******************************************************************
 * Delete a category name from the category being edited, handles
 * remapping the deleted category to 'none'.
 ******************************************************************/
void
EditCatEntryDeleteEntry(UInt deleteCatIdx)
{
  Byte idx;
  Byte zero;
  FormPtr formPtr;
  VoidHand catHand;
  TSCatRecType *catPtr;

  formPtr = FrmGetActiveForm();
  /* delete the specified category, adjusting category entries to retain alphabetic order */
  catHand = CatDeleteEntry(TSEditCatRecIdx, TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1], deleteCatIdx);
  catPtr = MemHandleLock(catHand);
  /* remap all deleted transTable entries to the 0, 'none' entry */
  zero = 0;
  for (idx = 1; idx < TSMaxCatEntries; idx += 1)
    {
      if (catPtr->transTable[idx] == 255)       /* marks the entries that have just been deleted */
        {
          /* remap entry to 0 'none' entry */
          DmWrite(catPtr, sizeof(Byte) * idx, &zero, sizeof(Byte));
        }
    }
  /* successfully deleted category */
  TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1] -= 1;
  MemHandleUnlock(catHand);
  // DmReleaseRecord(TSDatabase, TSEditCatRecIdx, true);
  /// [jo] error here as 'record not active'. Check out usage of TSEditCatRecIdx
  /// CatDeleteEntry has already released the record at [c]
  /// might we simply remove the CatDeleteEntry [c] line?
  /// Do we then have a problem as EditCatEntryRenameEntry also calls this fx?
  /// 
  /// tempting just to rem out the DmReleaseRecord above, but looks WRONG.
}

/******************************************************************
 * Modify an existing category entry. Returns -1 if the new category
 * name the user has entered is invalid, or 0 if the new category
 * name is the *same* as the existing name (the user has changed
 * nothing). If the category has been renamed successfully the index
 * of the
 ******************************************************************/
Int
EditCatEntryRenameEntry(void)
{
  CharPtr renamedCat;
  Int insertCatIdx;
  Byte idx;
  Byte catIdx;
  FormPtr formPtr;
  FieldPtr fieldPtr;
  VoidHand catHand;
  TSCatRecType *catPtr;

  formPtr = FrmGetActiveForm();
  fieldPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, fieldID_tsCat));
  /* check the user has actually entered a category name */
  renamedCat = FldGetTextPtr(fieldPtr);
  if (renamedCat == NULL || renamedCat[0] == NULL || IsAllSpace(renamedCat))
    {
      /* string appears to be empty, alert the user and return */
      FrmAlert(alertID_tsEmptyCat);
      return -1;
    }
  /* check the category name is actually different */
  catHand = DmGetRecord(TSDatabase, TSEditCatRecIdx);
  catPtr = MemHandleLock(catHand);
  if (StrCompare(renamedCat, catPtr->cats[TSEditCatEntryIdx]) == 0)
    {
      /* don't have to do anything, user has changed nothing ;) */
      MemHandleUnlock(catHand);
      DmReleaseRecord(TSDatabase, TSEditCatRecIdx, false);
      return 0;
    }
  /* check the category name isn't a special category name */
  if (StrCompare(renamedCat, catPtr->cats[0]) == 0 ||
      StrCompare(renamedCat, catPtr->cats[TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1] - 1]) == 0)
    {
      FrmAlert(alertID_tsReservedCat);
      MemHandleUnlock(catHand);
      DmReleaseRecord(TSDatabase, TSEditCatRecIdx, false);
      return -1;
    }
  /* delete the renamed entry (also modifies trans table) */
  MemHandleUnlock(catHand);
  DmReleaseRecord(TSDatabase, TSEditCatRecIdx, true);
  CatDeleteEntry(TSEditCatRecIdx, TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1], TSEditCatEntryIdx);
  /* insert the renamed entry into the entry array alphabetically */
  insertCatIdx = CatInsertEntry(TSEditCatRecIdx,
                                TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1] - 1 /* just deleted entry */ ,
                                renamedCat);
  catHand = DmGetRecord(TSDatabase, TSEditCatRecIdx);
  catPtr = MemHandleLock(catHand);
  if (insertCatIdx < 0)
    {
      /* category has been renamed to match an existing category which already exists */
      catIdx = -insertCatIdx - 1;       /* type conversion */
      /* map all deleted transTable entries to category that matches rename */
      for (idx = 1; idx < TSMaxCatEntries; idx += 1)
        {
          if (catPtr->transTable[idx] == 255)
            {
              DmWrite(catPtr, sizeof(Byte) * idx, &catIdx, sizeof(Byte));
            }
        }
      /* now have one less category entry */
      TSAppPrefs.numCatEntries[TSEditCatRecIdx - 1] -= 1;
    }
  else
    {
      /* category has been renamed to another unique category name, have inserted name, update trans table */
      catIdx = insertCatIdx;    /* type conversion */
      /* map all deleted transTable entries to category that matches new category name */
      for (idx = 1; idx < TSMaxCatEntries; idx += 1)
        {
          if (catPtr->transTable[idx] == 255)
            {
              DmWrite(catPtr, sizeof(Byte) * idx, &catIdx, sizeof(Byte));
            }
        }
    }
  /* successfully renamed category */
  MemHandleUnlock(catHand);
  DmReleaseRecord(TSDatabase, TSEditCatRecIdx, true);
  return catIdx;
}

/******************************************************************
 * Returns true if the passed string contains whitespace only.
 ******************************************************************/
Boolean
IsAllSpace(CharPtr s)
{
  while ((*s) != NULL)
    {
      if (!ISSPACE(*s))
        {
          return false;
        }
      s += 1;
    }
  return true;
}

/*** End of section! ***/



// ============================================================================ //
// PrefForm:
// #include "PrefForm.c"

/******************************************************************
 *
 ******************************************************************/
Boolean
PrefFormEventHandler(EventPtr event)
{
  FormPtr formPtr;
  Boolean handled = false;

  if (event->eType == ctlSelectEvent)
    {
      if (event->data.ctlEnter.controlID == buttonID_tsOk)
        {
          /* Save the potentially altered prefs */
          PrefFormSavePrefs(FrmGetActiveForm());
          /* Return to the previous form */
          FrmReturnToForm(formID_tsMain);
          /* Refresh the main display appearance as prefs may have changed */
          FormUpdateDay(&TSPresentDayDate);
          MainFormRefresh();
          handled = true;
        }
      else if (event->data.ctlEnter.controlID == buttonID_tsCancel)
        {
          /* return to the previous form */
          FrmReturnToForm(0);
          handled = true;
        }
    }
  else if (event->eType == frmOpenEvent)
    {
      /* the Pref form has been opened */
      formPtr = FrmGetActiveForm();
      PrefFormInit(formPtr);
      FrmDrawForm(formPtr);
      handled = true;
    }

  return handled;
}

/******************************************************************
 * Set up the prefs form depending on the current app prefs
 ******************************************************************/
void
PrefFormInit(FormPtr formPtr)
{
  ControlPtr controlPtr;

  /* set the appropriate group button for date format */
  FrmSetControlGroupSelection(formPtr, (UInt8) groupID_tsFormatGroup,
      (UInt16) buttonID_tsLongFormat + (TSAppPrefs.prefFlags & TSPrefShortDateFlag)); ///
  /* set the underline chargeable clients pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsUnderline));
  /* note below is correct, set if user does NOT want client underlining */
  CtlSetValue(controlPtr, TSAppPrefs.prefFlags & TSPrefUlineClientFlag);
  /* set the chargeable totals pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsChargeTotals));
  /* note below is correct, set if user does NOT want chargeable totals */
  CtlSetValue(controlPtr, !(TSAppPrefs.prefFlags & TSPrefChargeTotalsFlag));
  /* set the auto categories pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsAutoCats));
  CtlSetValue(controlPtr, (TSAppPrefs.prefFlags & TSPrefAutoCatsFlag));
  /* set the auto durations pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsAutoDur));
  CtlSetValue(controlPtr, (TSAppPrefs.prefFlags & TSPrefAutoDurFlag));
  /* set the default new entry chargeable pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsCharge));
  CtlSetValue(controlPtr, (TSAppPrefs.prefFlags & TSPrefDefaultChargeFlag));
  /* Set the keep timing pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsKeepTiming));
  CtlSetValue(controlPtr, (TSAppPrefs.prefFlags & TSPrefAlwaysTimeFlag));
}

/******************************************************************
 * Save the update prefs back to the global app preferences
 ******************************************************************/
void
PrefFormSavePrefs(FormPtr formPtr)
{
  ControlPtr controlPtr;

  /* save the date format selection */
  if ((UInt16) FrmGetObjectId(formPtr, (UInt16) FrmGetControlGroupSelection(formPtr, (UInt8)groupID_tsFormatGroup)) == buttonID_tsShortFormat) ///
    {
      /* set flag */
      TSAppPrefs.prefFlags |= TSPrefShortDateFlag;
    }
  else
    {
      /* clear flag */
      TSAppPrefs.prefFlags &= (~TSPrefShortDateFlag);
    }
  /* save the underline chargeable clients pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsUnderline));
  if (CtlGetValue(controlPtr))
    {
      /* set flag */
      TSAppPrefs.prefFlags |= TSPrefUlineClientFlag;
    }
  else
    {
      /* clear flag */
      TSAppPrefs.prefFlags &= (~TSPrefUlineClientFlag);
    }
  /* save the chargeable totals pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsChargeTotals));
  if (!CtlGetValue(controlPtr))
    {
      /* set flag */
      TSAppPrefs.prefFlags |= TSPrefChargeTotalsFlag;
    }
  else
    {
      /* clear flag */
      TSAppPrefs.prefFlags &= (~TSPrefChargeTotalsFlag);
    }    
  /* save the auto categories pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsAutoCats));
  if (CtlGetValue(controlPtr))
    {
      /* set flag */
      TSAppPrefs.prefFlags |= TSPrefAutoCatsFlag;
    }
  else
    {
      /* clear flag */
      TSAppPrefs.prefFlags &= (~TSPrefAutoCatsFlag);
    }
  /* save the auto durations pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsAutoDur));
  if (CtlGetValue(controlPtr))
    {
      /* set flag */
      TSAppPrefs.prefFlags |= TSPrefAutoDurFlag;
    }
  else
    {
      /* clear flag */
      TSAppPrefs.prefFlags &= (~TSPrefAutoDurFlag);
    }
  /* save the default new entry chargeable pref */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsCharge));
  if (CtlGetValue(controlPtr))
    {
      /* set flag */
      TSAppPrefs.prefFlags |= TSPrefDefaultChargeFlag;
    }
  else
    {
      /* clear flag */
      TSAppPrefs.prefFlags &= (~TSPrefDefaultChargeFlag);
    }
  /* save the default keep timing flag */
  controlPtr = FrmGetObjectPtr(formPtr, FrmGetObjectIndex(formPtr, checkboxID_tsKeepTiming));
  if (CtlGetValue(controlPtr))
    {
      /* set flag */
      TSAppPrefs.prefFlags |= TSPrefAlwaysTimeFlag;
    }
  else
    {
      /* clear flag */
      TSAppPrefs.prefFlags &= (~TSPrefAlwaysTimeFlag);
    }
}

/*** End of section! ***/

// ============================================================================ //
// AboutForm:
// #include "AboutForm.c"

/******************************************************************
 *
 ******************************************************************/
Boolean AboutFormEventHandler( EventPtr event )
{
  Boolean handled = false;

  if ( event->eType == ctlSelectEvent )
    {
      /*** Pen has been tapped on OK buttom, so leave the vanity form ***/
      FrmReturnToForm( 0 );
      handled = true;
    }
  else if ( event->eType == frmOpenEvent )
    {
      /*** The About form has been opened ***/
      FrmDrawForm( FrmGetActiveForm() );
      handled = true;
    }
  return handled;
}

/*** End of section! ***/


// ============================================================================ //
// DEBUGGERY: [jo]
// ============================================================================ //

void SHOWERROR(const Char * msg, Int32 nmbr)
{ 
  MemHandle memH=0;
  MemPtr memP=0;   
  
  memH = MemHandleNew(maxStrIToALen);
  if (! memH)
     { return;                                 // fail.
     };
  memP = MemHandleLock(memH);
  if (! memP)                                    //
     { return;                                 // fail.
     };  
  StrIToA((Char *)memP, nmbr);            // StrIToA uses Int32
  FrmCustomAlert (DebugAlert, msg, (Char *)memP, "Message:\n"); 

if (MemPtrUnlock (memP) != errNone)
     { return;                // BadErrUnlockTempPtr;
     };
  if (MemPtrFree (memP) != errNone)
     { return;                // BadErrTempFreePtr;
     };
  return;
} 

// ---------------------------------
// e_Delete: Free up memory on exit

Int16 e_Delete (MemPtr memP) 
{ if (MemPtrUnlock (memP) != errNone)
     { return 0;                // BadErrUnlockTempPtr;
     };
  if (MemPtrFree (memP) != errNone)
     { return 0;                // BadErrTempFreePtr;
     };
  return 1; // success
}

// ---------------------------------
MemPtr e_New ( Int16 memsize) 
{ 
  MemHandle memH=0;
  MemPtr memP=0;   
  memH = MemHandleNew(memsize);                  // *system* fx.
  if (! memH)
     { return 0;                                 // fail.
     };
  memP = MemHandleLock(memH);
  if (! memP)                                    //
     { return 0;                                 // fail.
     };
  return memP;                                   //pointer to locked new memory.
}

// ---------------------------------
Int16 xFill (Char * p0, Int16 slen, Char c)
{   // okay, there is a faster way. Check out PalmOS dox.
  while (slen > 0)
    { * p0 = c;
      p0 ++;
      slen --;
    };    
 return 1;           // success
} 

// ---------------------------------
Int16 xCopy (Char * dest, Char * xsrc, Int16 maxlen)
{ Int16 cnt;
  if (! dest)
     { return 0;
     };
  if (! xsrc)
     { return 0;
     };
  cnt = maxlen;
  while (cnt > 0)                                // permissible to copy NO bytes!
    { *dest++ = *xsrc++;
      cnt --;  // clumsy
    };
  return 1;   
}
// ============================================================================ //
//                            REALLY THE END!                                   //
// ============================================================================ //
